"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generatePredictionSequence = exports.generateTrainingSequence = void 0;

var _typings = require("../../typings");

const SLOTS_REGEX = /\[(.+?)\]\(([\w_\.-]+)\)/gi; // TODO replace this for appropriate tokenizer

const _tokenize = input => {
  return input.split(' ').filter(w => w.length);
};

const _makeToken = (value, matchedEntities, start, tag = '', slot = '') => {
  const token = {
    value,
    matchedEntities,
    start,
    end: start + value.length
  };

  if (tag) {
    token.tag = tag;
  }

  if (slot) {
    token.slot = slot;
  }

  return token;
}; // TODO use the same algorithm as in the prediction sequence


const _generateTrainingTokens = (input, start, slot = '', slotDefinitions = []) => {
  const slotDef = slotDefinitions.find(slotDef => !!slot && slotDef.name === slot);
  const matchedEntities = slotDef ? slotDef.entities : [];
  return _tokenize(input).map((t, idx) => {
    let tag = _typings.BIO.OUT;

    if (slot) {
      tag = idx === 0 ? _typings.BIO.BEGINNING : _typings.BIO.INSIDE;
    }

    const token = _makeToken(t, matchedEntities, start, tag, slot);

    start += t.length + 1; // 1 is the space char, replace this by what was done in the prediction sequence

    return token;
  });
};

const generateTrainingSequence = (input, slotDefinitions, intentName = '', contexts = []) => {
  let matches;
  let start = 0;
  let tokens = [];

  do {
    matches = SLOTS_REGEX.exec(input);

    if (matches) {
      const sub = input.substr(start, matches.index - start - 1);
      tokens = [...tokens, ..._generateTrainingTokens(sub, start), ..._generateTrainingTokens(matches[1], start + matches.index, matches[2], slotDefinitions)];
      start = matches.index + matches[0].length;
    }
  } while (matches);

  if (start !== input.length) {
    const lastingPart = input.substr(start, input.length - start);
    tokens = [...tokens, ..._generateTrainingTokens(lastingPart, start)];
  }

  return {
    intent: intentName,
    cannonical: tokens.map(t => t.value).join(' '),
    tokens,
    contexts
  };
};

exports.generateTrainingSequence = generateTrainingSequence;

const generatePredictionSequence = (input, intentName, entities) => {
  const cannonical = input; // we generate a copy here since input is mutating

  let currentIdx = 0;

  const tokens = _tokenize(input).map(value => {
    const inputIdx = input.indexOf(value);
    currentIdx += inputIdx; // in case of tokenization uses more than one char i.e words separated with multiple spaces

    input = input.slice(inputIdx + value.length);
    const matchedEntities = entities.filter(e => e.meta.start <= currentIdx && e.meta.end >= currentIdx + value.length).map(e => e.name);

    const token = _makeToken(value, matchedEntities, currentIdx);

    currentIdx = token.end; // move cursor to end of token in original input

    return token;
  });

  return {
    intent: intentName,
    cannonical,
    tokens
  };
};

exports.generatePredictionSequence = generatePredictionSequence;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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