"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SenarioRunner = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const TIMEOUT = 3000;

class SenarioRunner {
  constructor(bp) {
    this.bp = bp;

    _defineProperty(this, "_active", void 0);

    _defineProperty(this, "_status", void 0);

    _defineProperty(this, "_interval", void 0);

    _defineProperty(this, "_sendMessage", (message, eventDestination) => {
      setTimeout(() => {
        const event = this.bp.IO.Event({ ...eventDestination,
          direction: 'incoming',
          payload: {
            type: 'text',
            text: message
          },
          type: 'text'
        });
        this.bp.events.sendEvent(event);
      }, 1000);
    });

    this._active = [];
  }

  startReplay() {
    this._status = {};
    this._active = [];
    this._interval = setInterval(this._checkScenarioTimeout.bind(this), 5000);
  }

  processIncoming(event) {
    if (!this._active.length) {
      return;
    }

    const scenario = this._active.find(x => x.eventDestination.target === event.target);

    if (scenario && !scenario.completedSteps.length) {
      // The hook will replace the state with the one received here
      return scenario.initialState;
    }
  }

  processCompleted(event) {
    if (!this._active.length) {
      return;
    }

    const scenario = this._active.find(x => x.eventDestination.target === event.target);

    if (!scenario) {
      return;
    }

    const {
      name,
      completedSteps,
      steps
    } = scenario;
    const conversation = (0, _utils.convertLastMessages)(event.state.session.lastMessages, event.id);

    if (!conversation) {
      this._failScenario(name, {
        reason: 'Could not extract messages for the event ' + event.id
      });

      return;
    }

    const mismatch = this._findMismatch(steps[completedSteps.length], conversation);

    if (mismatch) {
      return this._failScenario(name, mismatch);
    } else {
      completedSteps.push(conversation);

      this._updateStatus(name, {
        completedSteps: completedSteps.length
      });
    }

    if (steps.length !== completedSteps.length) {
      scenario.lastEventTs = +new Date();

      this._sendMessage(steps[completedSteps.length].userMessage, scenario.eventDestination);
    } else {
      this._passScenario(name);
    }
  }

  runScenario(scenario, eventDestination) {
    const firstMessage = scenario.steps[0].userMessage;

    if (!firstMessage) {
      return;
    }

    this._active.push({ ...scenario,
      eventDestination,
      completedSteps: []
    });

    this._sendMessage(firstMessage, eventDestination);

    this._updateStatus(scenario.name, {
      status: 'pending',
      completedSteps: 0
    });
  }

  getStatus(scenarioName) {
    return this._status && this._status[scenarioName] || {};
  }

  isRunning() {
    return !!this._active.length;
  }

  _findMismatch(expected, received) {
    let mismatch = undefined; // This shouldn't happen

    if (!expected || !received || expected.userMessage !== received.userMessage) {
      return {
        reason: 'Expected or received step was invalid',
        expected,
        received
      };
    } // Inside each steps, the bot may reply multiple times


    _lodash.default.each(_lodash.default.zip(expected.botReplies, received.botReplies), ([exp, rec], idx) => {
      // This can happen if the bot doesn't respond
      if (!exp || !rec) {
        mismatch = {
          reason: 'Missing an expected or received reply',
          expected,
          received,
          index: idx
        };
        return false;
      }

      const sameSource = exp.replySource === rec.replySource;
      const sameResponse = exp.botResponse === rec.botResponse;
      const source = exp.replySource.split(' ').shift(); // extracting the first part (module) for the reply

      /**
       * Different sources are definitely not what is expected
       * If QNA has the exact same source, then we don't care about the response (variations)
       * If the source is Dialog Manager, then the answer must be identical (either payload or content element id)
       */

      if (!sameSource || source !== 'qna' && source === 'dialogManager' && !sameResponse) {
        mismatch = {
          reason: 'The reply was invalid',
          expected,
          received,
          index: idx
        };
        return false;
      }
    });

    return mismatch;
  }

  _checkScenarioTimeout() {
    if (!this._active.length) {
      this._interval && clearInterval(this._interval);
      return;
    }

    const now = +new Date();
    const mismatch = {
      reason: 'The scenario timed out'
    };

    this._active.filter(s => s.lastEventTs !== undefined && now - s.lastEventTs > TIMEOUT).map(x => this._failScenario(x.name, mismatch));
  }

  _passScenario(name) {
    this._updateStatus(name, {
      status: 'pass'
    });

    this._active = this._active.filter(x => x.name !== name);
  }

  _failScenario(name, mismatch) {
    this._updateStatus(name, {
      status: 'fail',
      mismatch
    });

    this._active = this._active.filter(x => x.name !== name);
  }

  _updateStatus(scenario, obj) {
    this._status[scenario] = { ...(this._status[scenario] || {}),
      ...obj
    };
  }

}

exports.SenarioRunner = SenarioRunner;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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