"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initModule = exports.initBot = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _nlu = _interopRequireWildcard(require("./providers/nlu"));

var _qnaMaker = _interopRequireDefault(require("./providers/qnaMaker"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const initBot = async (bp, botScopedStorage, botId) => {
  const config = await bp.config.getModuleConfigForBot('qna', botId);
  let storage = undefined;

  if (config.qnaMakerApiKey) {
    storage = new _qnaMaker.default(bp, config);
  } else {
    storage = new _nlu.default(bp, config, botId);
  }

  await storage.initialize();
  botScopedStorage.set(botId, storage);
};

exports.initBot = initBot;

const initModule = async (bp, botScopedStorage) => {
  bp.events.registerMiddleware({
    name: 'qna.incoming',
    direction: 'incoming',
    handler: async (event, next) => {
      if (!event.hasFlag(bp.IO.WellKnownFlags.SKIP_QNA_PROCESSING)) {
        const config = await bp.config.getModuleConfigForBot('qna', event.botId);
        const storage = botScopedStorage.get(event.botId);
        await processEvent(event, {
          bp,
          storage,
          config
        });
        next();
      }
    },
    order: 11,
    // must be after the NLU middleware and before the dialog middleware
    description: 'Listen for predefined questions and send canned responses.'
  });

  const getAlternativeAnswer = question => {
    const randomIndex = Math.floor(Math.random() * question.answers.length);
    return question.answers[randomIndex];
  };

  const buildSuggestions = async (event, question, confidence, intent, renderer) => {
    const payloads = [];

    if (question.action.includes('text')) {
      const args = {
        user: _lodash.default.get(event, 'state.user') || {},
        session: _lodash.default.get(event, 'state.session') || {},
        temp: _lodash.default.get(event, 'state.temp') || {},
        text: getAlternativeAnswer(question),
        typing: true
      };
      const element = await bp.cms.renderElement(renderer, args, {
        botId: event.botId,
        channel: event.channel,
        target: event.target,
        threadId: event.threadId
      });
      payloads.push(...element);
    }

    if (question.action.includes('redirect')) {
      payloads.push({
        type: 'redirect',
        flow: question.redirectFlow,
        node: question.redirectNode
      });
    }

    return {
      confidence,
      payloads,
      source: 'qna',
      sourceDetails: intent
    };
  };

  const getQuestionForIntent = async (storage, intentName) => {
    if (intentName && intentName.startsWith(_nlu.NLU_PREFIX)) {
      const qnaId = intentName.substring(_nlu.NLU_PREFIX.length);
      return (await storage.getQuestion(qnaId)).data;
    }
  };

  const processEvent = async (event, {
    bp,
    storage,
    config
  }) => {
    if (config.qnaMakerApiKey) {
      const qnaQuestion = (await storage.answersOn(event.preview)).pop();

      if (qnaQuestion && qnaQuestion.enabled) {
        event.suggestions.push((await buildSuggestions(event, qnaQuestion, qnaQuestion.confidence, undefined, config.textRenderer)));
      }

      return;
    }

    if (!event.nlu || !event.nlu.intents) {
      return;
    }

    for (const intent of event.nlu.intents) {
      const question = await getQuestionForIntent(storage, intent.name);

      if (question && question.enabled) {
        event.suggestions.push((await buildSuggestions(event, question, intent.confidence, intent.name, config.textRenderer)));
      }
    }
  };
};

exports.initModule = initModule;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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