"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _awsSdk = _interopRequireDefault(require("aws-sdk"));

var _lodash = _interopRequireDefault(require("lodash"));

var _moment = _interopRequireDefault(require("moment"));

var _multer = _interopRequireDefault(require("multer"));

var _multerS = _interopRequireDefault(require("multer-s3"));

var _path = _interopRequireDefault(require("path"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const ERR_USER_ID_REQ = '`userId` is required and must be valid';
const ERR_MSG_TYPE = '`type` is required and must be valid';
const ERR_CONV_ID_REQ = '`conversationId` is required and must be valid';

var _default = async (bp, db) => {
  const globalConfig = await bp.config.getModuleConfig('channel-web');

  const diskStorage = _multer.default.diskStorage({
    destination: globalConfig.fileUploadPath,
    limits: {
      files: 1,
      fileSize: 5242880 // 5MB

    },
    filename: function (req, file, cb) {
      const userId = _lodash.default.get(req, 'params.userId') || 'anonymous';

      const ext = _path.default.extname(file.originalname);

      cb(undefined, `${userId}_${new Date().getTime()}${ext}`);
    }
  });

  let upload = (0, _multer.default)({
    storage: diskStorage
  });

  if (globalConfig.uploadsUseS3) {
    /*
      You can override AWS's default settings here. Example:
      { region: 'us-east-1', apiVersion: '2014-10-01', credentials: {...} }
     */
    const awsConfig = {
      region: globalConfig.uploadsS3Region,
      credentials: {
        accessKeyId: globalConfig.uploadsS3AWSAccessKey,
        secretAccessKey: globalConfig.uploadsS3AWSAccessSecret
      }
    };

    if (!awsConfig.credentials.accessKeyId && !awsConfig.credentials.secretAccessKey) {
      delete awsConfig.credentials;
    }

    if (!awsConfig.region) {
      delete awsConfig.region;
    }

    const s3 = new _awsSdk.default.S3(awsConfig);
    const s3Storage = (0, _multerS.default)({
      s3: s3,
      bucket: globalConfig.uploadsS3Bucket || 'uploads',
      contentType: _multerS.default.AUTO_CONTENT_TYPE,
      cacheControl: 'max-age=31536000',
      // one year caching
      acl: 'public-read',
      key: function (req, file, cb) {
        const userId = _lodash.default.get(req, 'params.userId') || 'anonymous';

        const ext = _path.default.extname(file.originalname);

        cb(undefined, `${userId}_${new Date().getTime()}${ext}`);
      }
    });
    upload = (0, _multer.default)({
      storage: s3Storage
    });
  }

  const router = bp.http.createRouterForBot('channel-web', {
    checkAuthentication: false,
    enableJsonBodyParser: true
  });

  const asyncApi = fn => async (req, res, next) => {
    try {
      await fn(req, res, next);
    } catch (err) {
      bp.logger.attachError(err).error('HTTP Handling Error');
      res.status(500).send(err && err.message);
    }
  };

  router.get('/botInfo', asyncApi(async (req, res) => {
    const {
      botId
    } = req.params;
    const config = await bp.config.getModuleConfigForBot('channel-web', botId);
    const botInfo = await bp.bots.getBotById(botId);

    if (!botInfo) {
      return res.sendStatus(404);
    }

    res.send({
      showBotInfoPage: config.infoPage && config.infoPage.enabled || config.showBotInfoPage,
      name: botInfo.name,
      description: config.infoPage && config.infoPage.description || botInfo.description,
      details: botInfo.details
    });
  })); // ?conversationId=xxx (optional)

  router.post('/messages/:userId', bp.http.extractExternalToken, asyncApi(async (req, res) => {
    const {
      botId,
      userId = undefined
    } = req.params;

    if (!validateUserId(userId)) {
      return res.status(400).send(ERR_USER_ID_REQ);
    }

    const user = await bp.users.getOrCreateUser('web', userId);
    const payload = req.body || {};
    let {
      conversationId = undefined
    } = req.query || {};
    conversationId = conversationId && parseInt(conversationId);

    if (!_lodash.default.includes(['text', 'quick_reply', 'form', 'login_prompt', 'visit', 'request_start_conversation', 'postback'], payload.type)) {
      // TODO: Support files
      return res.status(400).send(ERR_MSG_TYPE);
    }

    if (payload.type === 'visit') {
      const {
        timezone,
        language
      } = payload;
      const isValidTimezone = _lodash.default.isNumber(timezone) && timezone >= -12 && timezone <= 14 && timezone % 0.5 === 0;
      const isValidLanguage = language.length < 4 && !_lodash.default.get(user, 'result.attributes.language');
      const newAttributes = { ...(isValidTimezone && {
          timezone
        }),
        ...(isValidLanguage && {
          language
        })
      };

      if (Object.getOwnPropertyNames(newAttributes).length) {
        await bp.users.updateAttributes('web', userId, newAttributes);
      }
    }

    if (!conversationId) {
      conversationId = await db.getOrCreateRecentConversation(botId, userId, {
        originatesFromUserMessage: true
      });
    }

    await sendNewMessage(botId, userId, conversationId, payload, req.credentials);
    return res.sendStatus(200);
  })); // ?conversationId=xxx (required)

  router.post('/messages/:userId/files', upload.single('file'), bp.http.extractExternalToken, asyncApi(async (req, res) => {
    const {
      botId = undefined,
      userId = undefined
    } = req.params || {};

    if (!validateUserId(userId)) {
      return res.status(400).send(ERR_USER_ID_REQ);
    }

    await bp.users.getOrCreateUser('web', userId); // Just to create the user if it doesn't exist

    let {
      conversationId = undefined
    } = req.query || {};
    conversationId = conversationId && parseInt(conversationId);

    if (!conversationId) {
      return res.status(400).send(ERR_CONV_ID_REQ);
    }

    const payload = {
      text: `Uploaded a file [${req.file.originalname}]`,
      type: 'file',
      storage: req.file.location ? 's3' : 'local',
      url: req.file.location || req.file.path || undefined,
      name: req.file.filename,
      originalName: req.file.originalname,
      mime: req.file.contentType || req.file.mimetype,
      size: req.file.size
    };
    await sendNewMessage(botId, userId, conversationId, payload, req.credentials);
    return res.sendStatus(200);
  }));
  router.get('/conversations/:userId/:conversationId', async (req, res) => {
    const {
      userId,
      conversationId,
      botId
    } = req.params;

    if (!validateUserId(userId)) {
      return res.status(400).send(ERR_USER_ID_REQ);
    }

    const conversation = await db.getConversation(userId, conversationId, botId);
    return res.send(conversation);
  });
  router.get('/conversations/:userId', async (req, res) => {
    const {
      botId = undefined,
      userId = undefined
    } = req.params || {};

    if (!validateUserId(userId)) {
      return res.status(400).send(ERR_USER_ID_REQ);
    }

    await bp.users.getOrCreateUser('web', userId);
    const conversations = await db.listConversations(userId, botId);
    const config = await bp.config.getModuleConfigForBot('channel-web', botId);
    return res.send({
      conversations: [...conversations],
      startNewConvoOnTimeout: config.startNewConvoOnTimeout,
      recentConversationLifetime: config.recentConversationLifetime
    });
  });

  function validateUserId(userId) {
    return /[a-z0-9-_]+/i.test(userId);
  }

  async function sendNewMessage(botId, userId, conversationId, payload, credentials) {
    const config = await bp.config.getModuleConfigForBot('channel-web', botId);

    if ((!payload.text || !_lodash.default.isString(payload.text) || payload.text.length > config.maxMessageLength) && payload.type != 'postback') {
      throw new Error('Text must be a valid string of less than 360 chars');
    }

    let sanitizedPayload = payload;

    if (payload.sensitive) {
      const sensitive = Array.isArray(payload.sensitive) ? payload.sensitive : [payload.sensitive];
      sanitizedPayload = _lodash.default.omit(payload, [...sensitive, 'sensitive']);
    }

    const event = bp.IO.Event({
      botId,
      channel: 'web',
      direction: 'incoming',
      payload,
      target: userId,
      threadId: conversationId,
      type: payload.type,
      credentials
    });
    const message = await db.appendUserMessage(botId, userId, conversationId, sanitizedPayload);
    bp.realtime.sendPayload(bp.RealTimePayload.forVisitor(userId, 'webchat.message', message));
    return bp.events.sendEvent(event);
  }

  router.post('/events/:userId', bp.http.extractExternalToken, asyncApi(async (req, res) => {
    const payload = req.body || {};
    const {
      botId = undefined,
      userId = undefined
    } = req.params || {};
    await bp.users.getOrCreateUser('web', userId);
    const conversationId = await db.getOrCreateRecentConversation(botId, userId, {
      originatesFromUserMessage: true
    });
    const event = bp.IO.Event({
      botId,
      channel: 'web',
      direction: 'incoming',
      target: userId,
      threadId: conversationId,
      type: payload.type,
      payload,
      credentials: req.credentials
    });
    bp.events.sendEvent(event);
    res.sendStatus(200);
  }));
  router.post('/conversations/:userId/:conversationId/reset', bp.http.extractExternalToken, asyncApi(async (req, res) => {
    const {
      botId,
      userId,
      conversationId
    } = req.params;
    await bp.users.getOrCreateUser('web', userId);
    const payload = {
      text: `Reset the conversation`,
      type: 'session_reset'
    };
    await sendNewMessage(botId, userId, conversationId, payload, req.credentials);
    const sessionId = await bp.dialog.createId({
      botId,
      target: userId,
      threadId: conversationId,
      channel: 'web'
    });
    await bp.dialog.deleteSession(sessionId);
    res.sendStatus(200);
  }));
  router.post('/conversations/:userId/new', async (req, res) => {
    const {
      userId,
      botId
    } = req.params;
    const convoId = await db.createConversation(botId, userId);
    res.send({
      convoId
    });
  });
  router.post('/conversations/:userId/:conversationId/reference/:reference', async (req, res) => {
    try {
      const {
        botId,
        userId,
        reference
      } = req.params;
      let {
        conversationId
      } = req.params;
      await bp.users.getOrCreateUser('web', userId);

      if (typeof reference !== 'string' || !reference.length || reference.indexOf('=') === -1) {
        throw new Error('Invalid reference');
      }

      if (!conversationId || conversationId == 'null') {
        conversationId = await db.getOrCreateRecentConversation(botId, userId, {
          originatesFromUserMessage: true
        });
      }

      const message = reference.slice(0, reference.lastIndexOf('='));
      const signature = reference.slice(reference.lastIndexOf('=') + 1);
      const verifySignature = await bp.security.getMessageSignature(message);

      if (verifySignature !== signature) {
        throw new Error('Bad reference signature');
      }

      const payload = {
        text: message,
        signature: signature,
        type: 'session_reference'
      };
      const event = bp.IO.Event({
        botId,
        channel: 'web',
        direction: 'incoming',
        target: userId,
        threadId: conversationId,
        type: payload.type,
        payload,
        credentials: req.credentials
      });
      await bp.events.sendEvent(event);
      res.sendStatus(200);
    } catch (error) {
      res.status(500).send({
        message: error.message
      });
    }
  });

  const getMessageContent = (message, type) => {
    const {
      payload
    } = message;

    if (type === 'file') {
      return payload && payload.url || message.message_data.url;
    } else if (type === 'text' || type === 'quick_reply') {
      return payload && payload.text || message.message_text;
    } else {
      return `Event (${type})`;
    }
  };

  const convertToTxtFile = async conversation => {
    const {
      messages
    } = conversation;
    const {
      result: user
    } = await bp.users.getOrCreateUser('web', conversation.userId);
    const timeFormat = 'MM/DD/YY HH:mm';
    const fullName = `${user.attributes['first_name'] || ''} ${user.attributes['last_name'] || ''}`;
    const metadata = `Title: ${conversation.title}\r\nCreated on: ${(0, _moment.default)(conversation.created_on).format(timeFormat)}\r\nUser: ${fullName}\r\n-----------------\r\n`;
    const messagesAsTxt = messages.map(message => {
      const type = message.payload && message.payload.type || message.message_type;

      if (type === 'session_reset') {
        return '';
      }

      const userName = message.full_name.indexOf('undefined') > -1 ? 'User' : message.full_name;
      return `[${(0, _moment.default)(message.sent_on).format(timeFormat)}] ${userName}: ${getMessageContent(message, type)}\r\n`;
    });
    return [metadata, ...messagesAsTxt].join('');
  };

  router.get('/conversations/:userId/:conversationId/download/txt', async (req, res) => {
    const {
      userId,
      conversationId,
      botId
    } = req.params;

    if (!validateUserId(userId)) {
      return res.status(400).send(ERR_USER_ID_REQ);
    }

    const conversation = await db.getConversation(userId, conversationId, botId);
    const txt = await convertToTxtFile(conversation);
    res.send({
      txt,
      name: `${conversation.title}.txt`
    });
  });
};

exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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