"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _bluebird = _interopRequireDefault(require("bluebird"));

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class HitlDb {
  constructor(bp) {
    this.bp = bp;

    _defineProperty(this, "knex", void 0);

    _defineProperty(this, "createUserSession", async event => {
      let profileUrl = undefined;
      let full_name = '#' + Math.random().toString().substr(2);
      const user = (await this.bp.users.getOrCreateUser(event.channel, event.target)).result;

      if (user && user.attributes && user.attributes.first_name && user.attributes.last_name) {
        profileUrl = user.attributes.profile_pic || user.attributes.picture_url;
        full_name = user.attributes.first_name + ' ' + user.attributes.last_name;
      }

      const session = {
        botId: event.botId,
        channel: event.channel,
        userId: event.target,
        user_image_url: profileUrl,
        last_event_on: this.knex.date.now(),
        last_heard_on: this.knex.date.now(),
        paused: 0,
        full_name: full_name,
        paused_trigger: undefined
      };
      const dbSession = await this.knex.insertAndRetrieve('hitl_sessions', session, '*');
      return {
        is_new_session: true,
        ...dbSession
      };
    });

    _defineProperty(this, "buildUpdate", direction => {
      const now = this.knex.date.now();
      return direction === 'in' ? {
        last_event_on: now
      } : {
        last_event_on: now,
        last_heard_on: now
      };
    });

    this.knex = bp.database;
  }

  initialize() {
    if (!this.knex) {
      throw new Error('you must initialize the database before');
    }

    return this.knex.createTableIfNotExists('hitl_sessions', function (table) {
      table.increments('id').primary();
      table.string('botId').notNullable();
      table.string('channel');
      table.string('userId');
      table.string('full_name');
      table.string('user_image_url');
      table.timestamp('last_event_on');
      table.timestamp('last_heard_on');
      table.boolean('paused');
      table.string('paused_trigger');
    }).then(() => {
      return this.knex.createTableIfNotExists('hitl_messages', function (table) {
        table.increments('id').primary();
        table.integer('session_id').references('hitl_sessions.id').onDelete('CASCADE');
        table.string('type');
        table.string('text', 640);
        table.jsonb('raw_message');
        table.enu('direction', ['in', 'out']);
        table.timestamp('ts');
      });
    }).then(() => this.knex('hitl_messages').columnInfo('text').then(info => {
      if (info.maxLength === null || this.knex.isLite) {
        return;
      }

      return this.knex.schema.alterTable('hitl_messages', table => {
        table.text('text', 'longtext').alter();
      });
    }));
  }

  async getOrCreateUserSession(event) {
    if (!event.target) {
      return undefined;
    }

    return this.knex('hitl_sessions').where({
      botId: event.botId,
      channel: event.channel,
      userId: event.target
    }).select('*').limit(1).then(users => {
      if (!users || users.length === 0) {
        return this.createUserSession(event);
      } else {
        return users[0];
      }
    });
  }

  getSessionById(sessionId) {
    return this.knex('hitl_sessions').where({
      id: sessionId
    }).select('*').limit(1).then(users => {
      if (!users || users.length === 0) {
        return undefined;
      } else {
        return users[0];
      }
    });
  }

  toPlainObject(object) {
    // trims SQL queries from objects
    return _lodash.default.mapValues(object, v => {
      return v && v.sql ? v.sql : v;
    });
  }

  appendMessageToSession(event, sessionId, direction) {
    const message = {
      session_id: sessionId,
      type: event.type,
      text: event.payload.text,
      raw_message: event.payload,
      direction: direction,
      ts: this.knex.date.now()
    };
    return _bluebird.default.join(this.knex('hitl_messages').insert(message), this.knex('hitl_sessions').where({
      id: sessionId
    }).update(this.buildUpdate(direction)), () => this.toPlainObject(message));
  }

  setSessionPaused(paused, session, trigger) {
    const {
      botId = undefined,
      channel = undefined,
      userId = undefined,
      sessionId = undefined
    } = session;

    if (sessionId) {
      return this.knex('hitl_sessions').where({
        id: sessionId
      }).update({
        paused: paused ? 1 : 0,
        paused_trigger: trigger
      }).then(() => parseInt(sessionId));
    } else {
      return this.knex('hitl_sessions').where({
        botId,
        channel,
        userId
      }).update({
        paused: paused ? 1 : 0,
        paused_trigger: trigger
      }).then(() => {
        return this.knex('hitl_sessions').where({
          botId,
          channel,
          userId
        }).select('id');
      }).then(sessions => parseInt(sessions[0].id));
    }
  }

  isSessionPaused(session) {
    const {
      botId = undefined,
      channel = undefined,
      userId = undefined,
      sessionId = undefined
    } = session;

    const toBool = s => this.knex.bool.parse(s);

    if (sessionId) {
      return this.knex('hitl_sessions').where({
        id: sessionId
      }).select('paused').then().get(0).then(s => s && toBool(s.paused));
    } else {
      return this.knex('hitl_sessions').where({
        botId,
        channel,
        userId
      }).select('paused').then().get(0).then(s => s && toBool(s.paused));
    }
  }

  getAllSessions(onlyPaused, botId) {
    let condition = '';
    const knex2 = this.knex;

    if (onlyPaused === true) {
      condition = 'hitl_sessions.paused = ' + this.knex.bool.true();
    }

    return this.knex.select('*').from(function () {
      this.select([knex2.raw('max(id) as mId'), 'session_id', knex2.raw('count(*) as count')]).from('hitl_messages').groupBy('session_id').as('q1');
    }).join('hitl_messages', this.knex.raw('q1.mId'), 'hitl_messages.id').join('hitl_sessions', this.knex.raw('q1.session_id'), 'hitl_sessions.id').whereRaw(condition).where({
      botId
    }).orderBy('hitl_sessions.last_event_on', 'desc').limit(100).then(results => ({
      total: 0,
      sessions: results
    }));
  }

  getSessionData(sessionId) {
    return this.knex('hitl_sessions').where({
      session_id: sessionId
    }).join('hitl_messages', 'hitl_messages.session_id', 'hitl_sessions.id').orderBy('hitl_messages.id', 'desc').limit(100).select('*').then(messages => _lodash.default.orderBy(messages, ['id'], ['asc']));
  }

}

exports.default = HitlDb;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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