package AutomateSchedule::Install;

use strict;
use LWP::UserAgent;
use File::Path;
use File::List;
use File::HomeDir;
use File::Basename;
use Cwd;
use Carp;
use Number::Bytes::Human qw(format_bytes);

my $PACKAGE;
my $self;
my $os;
my $osversion;
my $bitWidth;
my $rawArch;
my $prettyArch;
my $buildInfo;
my $whoami;
my $automateUser;
my $automateScheduleDir;  # where the automate schedule server should go
my $automateScheduleHomeDir;  # the automate user's home dir
my $javaHome;
my $java16 = "6";
my $java17 = "7";
my $java18 = "8";
my $useJavaVersion = $java17; # 6/7/8
my $userHome;  # the home dir of the user currently running this
my $userAgent;
my $startingDir;
my $fileDir;
my $postgresqlPossible;
my $DEFAULT_AUTOMATE_SCHEDULE_DOWNLOAD_URL = "http://download.helpsystems.com/download/";
my $DEFAULT_S3_DOWNLOAD_URL = "http://s3.amazonaws.com/helpsystems/download/";
# the agent installer may choose to point this at the Automate Schedule Server
my $automateScheduleDownloadURL;

sub new {

    $PACKAGE = shift;
    $fileDir = shift;
    if (! defined($fileDir)) {
        die("Missing file-dir parameter");
    }
    if (! -r $fileDir) {
        die("Directory $fileDir not found or inaccessible");
    }
    $self = bless({}, $PACKAGE);
    $self->resolvePlatform();
    $startingDir = cwd();
    $buildInfo = $self->readFileHash("$fileDir/buildinfo.txt");
    $userAgent = LWP::UserAgent->new;
    $userAgent->timeout(5);
    return $self;
}

sub resolvePlatform {
    # resolve the OS
    chomp($os = `uname`);
    chomp($osversion = `uname -r`);

    # determine what the machine thinks its architecture is.
    if ( ($os eq 'Linux') || ($os eq 'HP-UX') ) {
        chomp($rawArch = `uname -m`);
    } else {
        chomp($rawArch = `uname -p`);
    }

    if ( ($os eq 'HP-UX') && ($rawArch =~ m/^9000/) ) {
        $rawArch = '9000';
    }
    
    if ($os eq 'Darwin') {
        # Java 1.6 and 1.7 on Mac OSX is 64 bit only
        $rawArch="x86_64";
        chomp($osversion = `sw_vers | grep ProductVersion | cut -f2 -d:`);
        $osversion =~ s/^\s+//;
    }

    # assume these are the same - for now
    $prettyArch = $rawArch;

    # makes a best-guess at the architecture's bit width: 32 or 64 bit.
    $bitWidth = '32';
    # figure out the bit-width for AIX/powerpc
    if ( ($os eq 'AIX') && ($rawArch eq 'powerpc') ) {
        chomp($bitWidth = `prtconf -k | cut -c14-15`);
        if ('64' eq $bitWidth) {
            $prettyArch = $prettyArch . $bitWidth;
        }
        # on AIX, -r returns the sub/revision, -v returns the version.
        my $osSubVersion = $osversion;
        chomp($osversion = `uname -v`);
        $osversion = "$osversion.$osSubVersion";
    }

    # figure out the bit-width for Solaris/sparc
    if ($os eq 'SunOS') {
        chomp($bitWidth = `isainfo -b`);
        if ('64' eq $bitWidth) {
            if ('i386' eq $rawArch) {
                $rawArch = 'x86_64';
                $prettyArch = $rawArch;
            } else {
                # sparc becomes sparc64
                $prettyArch = $prettyArch . $bitWidth;
            }
        }
    }

    if ( ($os eq 'HP-UX') || ($rawArch eq 'x86_64') || 
        ($rawArch eq 's390x') || ($rawArch eq 'ppc64') || ($rawArch eq 'ppc64le') ) {
        # these architecture are inherently 64-bit.
        $bitWidth = '64';
    }

    chomp($whoami = `whoami`);
    if ( $whoami eq '') {
        chomp($whoami = `/usr/ucb/whoami`);
    }

    if ( $os eq "Darwin" ) {
        my @info = `dscacheutil -q user -a name automate`;
        foreach my $line (@info) {
            chomp($line);
            my ($key, $value) = split /: /, $line;
            if ('dir' eq $key) {
                $automateScheduleDir = $value;
            } 
            if ('name' eq $key) {
                $automateUser = $value;
            } 
        }
    } else {
        my $automateUserInfo = `grep -i ^automate: /etc/passwd`;
        if ( length($automateUserInfo) == 0 ) {
            # try NIS
            my $ypw='yp.pw';
            if ( -f $ypw) {
                system("rm $ypw") == 0 || die("Unable to remove temp file $ypw");
            }
            system("ypcat passwd > $ypw 2> /dev/null");
            if ( -f $ypw) {
                $automateUserInfo = `grep -i "^automate:" $ypw`;
            }
            system("rm $ypw");
        }
        if ( length($automateUserInfo) > 0) {
            my ($user, $pw, $uid, $gid, $ignored, $dir, $shell) = split /:/, $automateUserInfo;
            $automateUser = $user;
            $automateScheduleDir = $dir;
        }
    }

    $automateScheduleHomeDir = $automateScheduleDir;
    
    my $home = '^/home/';
    if (defined($automateScheduleDir) && ($automateScheduleDir =~ m/$home/)
        && -e '/opt/automate-schedule' ) {
        # on older Automate Schedule installs, the home dir may not point to the
        # automate schedule install dir. Check /opt for automate schedule products
        $automateScheduleDir = '/opt/automate-schedule';
    }

    # the current user's home dir, regardless of whether automate is present.
    $userHome = File::HomeDir->my_home;

    #  need to allow java 1.6 on the following because 1.7 is not available for them:
    # Sun OS Solaris 8 and 9 and 10.0-10.8
    # AIX 5.3
    # HP-UX PA-RISC
    # RH 4
    # Linux Enterprise (SUSE lower than 10 SP2; Red Hat 5.0-5.4
    # Apple OSX 10.6, or lower than 10.7.3 or 10.8.3

    if ($os eq 'SunOS') {

        $useJavaVersion = $java18;

        # Solaris 'version' 10.9 or higher is req. for Java 7
        # of course that is easier said than figued out!
        # uname -v on sparc returns Kernel version - yay!
        # uname -v on i386 returns the Kernel Patch ID, something like "Generic_147441-01"

        # osversion 5.8 and 5.9 (Solaris 8 and 9) use Java 6
        #debug lines
        #$osversion = 5.8;
        if ( ($osversion == 5.8) || ($osversion == 5.9) ) {
            $useJavaVersion = $java16;
        }
        # osversion 5.10 with KernelID (osSubversion) 10.1 - 10.8 use Java 6
        if ($osversion == 5.10) {
            #debug lines
            #$rawArch = 'sparc';
            if ($rawArch eq 'sparc') {
                my $osSubversion;
                chomp($osSubversion = `uname -v`);
                #debug lines
                #chomp($osSubversion = 10.3);
                my @pieces = split / /, $osSubversion;
                my $numberOfPieces = @pieces;
                my $osSubversion = $pieces[$numberOfPieces-1];
                # turn a number like 3.1.4 into major=3, minor=1, unwanted gets the rest
                my ($major, $minor, $unwanted) = split /\./, $osSubversion;
                #debug lines
                #print("from Install.pm: Sparc KernelID= $major.$minor\n");
                if ($minor < 9)  {
                    $useJavaVersion = $java16;
                }
            } else {
        # must be i386 - check for Kernel Patch ID (10.9 = 142910-17)
                my $solarisKernelID;
                chomp($solarisKernelID = `uname -v`);
                #debug lines
                #chomp($solarisKernelID = "Generic_141234-01");
                $solarisKernelID =~ m/Generic_(.*?)-/;
                print("from Install.pm: solarisKernelID = $solarisKernelID\n");
                $solarisKernelID = substr $1, 0 , 6;
                print("from Install.pm: solarisKernelID = $solarisKernelID\n");
                if ($solarisKernelID < 142910) {
                    $useJavaVersion = $java16;
                }
            }
        }
    }
    elsif ( ($os eq 'AIX') && ($rawArch eq 'powerpc') ) {
        $useJavaVersion = $java17;
        my $aix_oslevel;
        chomp ($aix_oslevel = `oslevel -s`);

        # turn a number like "6100-05-09-1228" into ver=6100, major=05, minor=09, unwanted gets the rest
        my ($ver, $major, $minor, $unwanted) = split /\-/, $aix_oslevel;
        # debug lines
        # $ver = 6100;
        # $major = 5;
        # $minor = 9;
        # $unwanted = 1228;
        # print("osversion, aix_oslevel result =$osversion, $ver.$major.$minor.$unwanted\n");

        if ( ($osversion == 5.3) || ( ($osversion == 6.1) && ($major <= 4) ) ) {
            $useJavaVersion = $java16;
        }
        elsif ( (($osversion == 6.1) && ($major >= 7)) ||
                (($osversion == 7.1) && ($major >= 3)) ||
                ($osversion >= 7.2) ) {
            $useJavaVersion = $java18;
        }
    }
    elsif ($os eq 'HP-UX') {
        if ($rawArch =~ m/^9000/) {
            $useJavaVersion = $java16;
        }
        else {
            my $hpuxVersion = "";
            chomp($hpuxVersion = `uname -r`);
            # Example: $hpuxVersion = B.11.31
            my ($unwanted, $major, $minor) = split /\./, $hpuxVersion;
            if ( ( ($major == 11) && ($minor >= 31) ) || ($major >= 12) ) {
                $useJavaVersion = $java18;
            }
        }
    }
    elsif ($os eq 'Linux') {
        # first check for glibc 2.4 or higher - if lower then java 1.7 won't work
        my $libcVersionString;
        chomp ($libcVersionString = `ldd --version | head -1`);
        #debug lines
        #chomp($libcVersionString="GNU C Library stable release version 2.3");
        # the last piece of the entire version string is *usually* the LIBC version.
        my @pieces = split / /, $libcVersionString;
        my $numberOfPieces = @pieces;
        my $libcVersion = $pieces[$numberOfPieces-1];
        # turn a number like 3.1.4 into major=3, minor=1, unwanted gets the rest
        my ($major, $minor, $unwanted) = split /\./, $libcVersion;
        # debug lines
        #    print("GNU LIBC result= $major.$minor\n");
        if (! ( ($major >= 2) || (($major == 2) && ($minor >= 4)) ) ) {
            $useJavaVersion = $java16;
        }
        if ( ($major > 2) || (($major == 2) && ($minor >= 12)) ) {
            $postgresqlPossible = 'true';
        } else {
            $postgresqlPossible = 'false';
            print("GNU LIBC version $major.$minor\n");
        }
        
        # glibc check might pass, but there certain mimimum versions for JRE7:
        # Oracle 5.5+
        # Red Hat 5.5+
        # Suse 10 SP2 (we are going to ignore SP level -- 10 or higher is fine)
        # Ubuntu 10.04+, 11.04+, 12.04+ (we are going to ingore sub-versions - assume ok if 10 or higher)
        if ($useJavaVersion != $java16) {
            my $ora_release = '/etc/oracle-release'; 
            my $rh_release  = '/etc/redhat-release';
            my $sus_release = '/etc/SuSE-release';
            my $ubu_release = '/etc/lsb-release';
            my $os_release = '/etc/os-release';

            # Keith 5/11/2015 - "lsb_release" does not appear to be reliably supported...
            my $lsb_release;
            chomp($lsb_release = `lsb_release -a 2> /dev/null | grep Release | cut -f2 -d:`);
            $lsb_release =~ s/^\s+//;
            $lsb_release =~ s/\s+$//;

            # Is this a Oracle or Red Hat Linux (or a derivative thereof)
            if ( ( -f $ora_release ) || ( -f $rh_release ) ) {

                if ($lsb_release == "") {
                    if (-f $rh_release) {
                        # print("Checking $rh_release for current release...\n");
                        my $rh_release_text = "";

                        # Example: "CentOS release 5.11 (Final)"
                        if ($lsb_release == "") {
                            # print("Checking for Centos release...\n");
                            chomp($lsb_release = `cat /etc/redhat-release |gawk '{print toupper(\$0)}'| grep "CENTOS RELEASE" | cut -f3 '-d '`);
                        }

                        # Example: "CentOS Linux release 7.0.1406 (Final)"
                        if ($lsb_release == "") {
                            # print("Checking for Centos 7+ release...\n");
                            chomp($lsb_release = `cat /etc/redhat-release |gawk '{print toupper(\$0)}'| grep "CENTOS LINUX RELEASE" | cut -f4 '-d '`);
                        }

                        # Example: "Red Hat Enterprise Linux Server release 6.12 (Final)"
                        if ($lsb_release == "") {
                            # print("Checking for Red Hat Enterprise Linux Server release...\n");
                            chomp($lsb_release = `cat /etc/redhat-release |gawk '{print toupper(\$0)}'| grep "RED HAT ENTERPRISE LINUX SERVER RELEASE" | cut -f7 '-d '`);
                        }

                        if ($lsb_release == "") {
                            print("Linux Red Hat/Oracle release is not available. Defaulting with 0.0.0.\n");
                            $lsb_release = "0.0.0"
                        }
                    }
                }

                if (-f $rh_release) {
                    my $fedora_release = "";
                    # print("Checking for Fedora release...\n");
                    # Example: "Fedora release 21 (Final)"
                    chomp($fedora_release = `cat /etc/redhat-release |gawk '{print toupper(\$0)}'| grep "FEDORA RELEASE" | cut -f3 '-d '`);
                    if (! ($fedora_release == "")) {
                        $lsb_release = "0.0.0";
                        print("Fedora detected...\n");
                    }
                }

                # turn a number like 5.4.3 into major=5, minor=4, unwanted gets the rest
                # print("Using Red Hat/Oracle Linux Release: $lsb_release\n");
                my ($major, $minor, $unwanted) = split /\./, $lsb_release;

                # $major = 5;
                # $minor = 4;
                # print("Oracle / Red Hat Linux - lsb_release major.minor result= $major.$minor\n");

                if ($major == 0) {
                    $useJavaVersion = $java17;
                }
                elsif ( ($major < 5) || (($major == 5) && ($minor <= 4)) ) {
                    $useJavaVersion = $java16;
                }
                elsif (($major == 5) && ($minor <= 10)) {
                    $useJavaVersion = $java17;
                }
                elsif ($major >= 5) {
                    $useJavaVersion = $java18;
                }
            }

            # Is this a Suse Linux
            elsif ( -f $sus_release ) {

                if ($lsb_release == "") {
                    # print("lsb_release for Suse Linux is not available (Defaulting with 0).\n");
                    $lsb_release = "0"
                }

                # print("Using Suse Linux Release: $lsb_release\n");

                # debug lines
                # $lsb_release = 9;
                # print("Suse Linux - lsb_release result= $lsb_release\n");

                if ($lsb_release == 0) {
                    $useJavaVersion = $java17;
                }
                elsif ($lsb_release < 10) {
                    $useJavaVersion = $java16;
                }
                elsif ($lsb_release == 10) {
                    $useJavaVersion = $java17;
                }
                elsif ($lsb_release >= 11) {
                    $useJavaVersion = $java18;
                }
            }

            # Is this Ubuntu Linux
            elsif ( -f $ubu_release ) {

                if ($lsb_release == "") {
                    # print("lsb_release for Ubuntu Linux is not available (Defaulting with 0.0.0).\n");
                    $lsb_release = "0.0.0"
                }

                # turn a number like 3.1.4 into major=3, minor=1, unwanted gets the rest
                # print("Using Ubuntu Linux Release: $lsb_release\n");
                my ($major, $minor, $unwanted) = split /\./, $lsb_release;

                # debug lines
                # $major = 5;
                # print("Ubuntu Linux - lsb_release major result= $major\n");

                if ($major == 0) {
                    $useJavaVersion = $java17;
                }
                elsif ($major < 10) {
                    $useJavaVersion = $java16;
                }
                elsif (($major == 10) || ($major == 11)) {
                    $useJavaVersion = $java17;
                }
                elsif ($major >= 12) {
                    $useJavaVersion = $java18;
                }
            }
            
            # if we did not find one of the distro files, check this one
            # SLES has deprecated suse-release, so we are checking here now
            elsif ( -f $os_release ) {
                my $os_name = `. /etc/os-release ; echo \$ID;`;
                my $os_ver = `. /etc/os-release ; echo \$VERSION_ID;`;
                
                if ( $os_name == "sles" && $os_ver >= 12) {
                    $useJavaVersion = $java18;
                }
                elsif ( $os_name == "debian" && $os_ver >= 8 ) {
                    $useJavaVersion = $java18;
                }
            }
            # We don't know what Linux OS this is - default to JRE7
            else {
                # debug lines
                # print("Not sure what type of Linux this is - put down JRE7\n");
                $useJavaVersion = $java17;
            }
        }

############################################################################################
# B:15362 - remove SELINUX checks.  Java 7 Updates should deprecate this requirement.
############################################################################################
##        # might also need to check if SELINUX is "enforcing" and put down JRE6 in that case
##        # sestatus on Enterprise Linux will return what I need
##        if ($useJavaVersion != $java16) {
##            my $selinux;
##            chomp($selinux = `/usr/sbin/sestatus 2> /dev/null | grep mode | cut -f2 -d:`);
##            $selinux =~ s/^\s+//;
##            $selinux =~ s/\s+$//;
##            # debug lines
##            # $selinux = 'permissive';
##            # print("selinux result= $selinux\n");
##            if ($selinux eq 'enforcing') {
##                print("Warning only - SELINUX enforcing detected. Reverting to use Java 1.6.\n");
##                $useJavaVersion = $java16;
##            }
##        }

        # End of Linux section - if JRE7+ will be put down, run 'execstack -c <libfile>'
        # note the server execstack is taken care of in installServer.pl
        if ( ($useJavaVersion != $java16) && ($fileDir eq 'files-agent') ) {
            # debug lines
            # print("path= files-agent/bin/Linux/$rawArch/libnativeSz.so\n");
            # print("execstack -c files-agent/bin/Linux/$rawArch/libnativeSz.so\n");
            my $result = system("execstack -c files-agent/bin/Linux/$rawArch/libnativeSz.so");
            if ($result > 0) {
                print("Warning only - Unable to mark nativeSz as not requiring an executable stack\n");
            }
        }
    }
    elsif ( ($os eq 'Darwin') && ($rawArch eq 'x86_64') ) {
        # turn a number like 10.7.3 into major=7, minor=3, unwanted gets the rest
        my ($unwanted, $major, $minor) = split /\./, $osversion;
        # debug lines
        # $major = 7;
        # $minor = 1;
        # print("osversion result= $major.$minor\n");
        if (! ((($major == 7) && ($minor >= 3)) || (($major == 8) && ($minor >= 3)) || ($major > 8)) ) {
            $useJavaVersion = $java16;
        }
        elsif (($major == 8) && ($minor >= 3)) {
            $useJavaVersion = $java17;
        }
        elsif ($major >= 9) {
            $useJavaVersion = $java18;
        }
    }
    else {
        $useJavaVersion = $java17;
    }
    # debug lines
    # print("from Install.pm: useJavaVersion (6/7/8)= $useJavaVersion\n");
}

sub setAutomateUser {
    my $self = shift;
    $automateUser = shift;
}

sub setAutomateScheduleDir {
    my $self = shift;
    $automateScheduleDir = shift;
    $automateScheduleHomeDir = $automateScheduleDir;
}

sub setUserHome {
    my $self = shift;
    $userHome = shift;
}

sub setAutomateScheduleDownloadURL {
    my $self = shift;
    $automateScheduleDownloadURL = shift;
}

sub os {
    return $os;
}

sub osVersion {
    return $osversion;
}

sub rawArch {
    return $rawArch;
}

sub arch {
    return $prettyArch;
}

sub platform {
    return "$os/$prettyArch";
}

sub bitWidth {
    return $bitWidth;
}

sub whoami {
    return $whoami;
}

sub automateUser {
    return $automateUser;
}

sub automateScheduleDir {
    return $automateScheduleDir;
}

sub automateScheduleHomeDir {
    return $automateScheduleHomeDir;
}

sub userHome {
    return $userHome;
}

sub useJavaVersion {
    return $useJavaVersion;
}

sub postgresqlPossible {
    return $postgresqlPossible;
}

sub buildInfo {
    return $buildInfo->{'automate_schedule_version'} 
            . ' build ' .  $buildInfo->{'automate_schedule_build_date'}
            . '-' . $buildInfo->{'automate_schedule_build_time'};
}

sub findJava {
    my $self = shift;

    if ( defined($javaHome) ) {
        return $javaHome;
    }

    my $javaDir;
    # look for a 'jre' dir in the automate-schedule dir
    if (defined($automateScheduleDir)) {
        my $javaDir = "$automateScheduleDir/jre";
        if ( -x "$javaDir/bin/java" ) {
            $javaHome = $javaDir;
            return $javaHome;
        }

        # look in the scripts for a definition of JAVA_HOME
        $javaDir = $self->searchForJavaHome(
            "$automateScheduleDir/server/startscheduler.sh",
            "$automateScheduleDir/scheduler/startscheduler.sh",
            "$automateScheduleDir/agent/start.sh");
        if (defined($javaDir)) {
            $javaHome = $javaDir;
            return $javaHome;
        }
    }
    
    # see if the JAVA_HOME env var is defined
    $javaDir = $ENV{'JAVA_HOME'};
    if ( -x "$javaDir/bin/java" ) {
        if ($self->testJava($javaDir)) {
            if ( length($javaDir) == 0 ) {
                $javaDir = '/';
            }
            $javaHome = $javaDir;
        }
    }

    #debug testing
    #print("Install.pm :: osversion is $osversion.\n");
    #$osversion = 10.6;
    #print("Install.pm :: fake osversion is $osversion.\n");

    if ( ($os eq 'Darwin') && ($rawArch eq 'x86_64') && ($osversion >= 10.6 ) && ($osversion < 10.7 ) ) {
        my $MAC_OSX_JAVA = "/System/Library/Frameworks/JavaVM.framework/Versions/1.6.0/Home";
        if (!defined($javaHome) && -e $MAC_OSX_JAVA) {
            if ($self->testJava($MAC_OSX_JAVA)) {
                $javaHome = $MAC_OSX_JAVA;
            }
        }
    }
    #debug lines
    # print("from sub findJava: javaHome = $javaHome\n\n");
    return $javaHome;
}

sub searchForJavaHome {
    my $self = shift;
    foreach my $argv (@_) {
        if (-r $argv) {
            my $line = `grep JAVA_HOME= $argv`;
            chomp($line);
            my ($ignored, $javaHome) = split /=/, $line;
            if ( -d $javaHome ) {
                return $javaHome;
            }
        }
    }
    return undef;
}

sub testJava {
    my $self = shift;
    my $javaDir = shift;

    my $javaExec = "$javaDir/bin/java";
    my $javaFlags = '';
    if ($os eq 'SunOS' && $bitWidth eq '64') {
        $javaFlags = '-d64';
    }
    if (! -x $javaExec ) {
        print("Unable to run java in $javaExec\n");
        return 0;
    }

    my $line = `$javaExec $javaFlags -version 2>&1 | head -1`;
    # debug lines
    #    print("=======================================\n");
    #    print("Here are the details about my testJava routine from Install.pm: \n");
    #    print("javaExec = $javaExec\n");
    #    print("javaFlags = $javaFlags\n");
    #    print("line = $line\n");
    #    print("useJavaVersion (6/7/8) = $useJavaVersion\n");
    #    print("=======================================\n\n");

    chomp($line);

    if ( ! (($line =~ m/1\.6/) || ($line =~ m/1\.7/) || ($line =~ m/1\.8/)) ){
        print("The Java environment '$javaExec' is not usable,\n   v1.6 or later required: $line\n");
        return 0;
    }

    my $javaTestJar = "$startingDir/$fileDir/javatests.jar";
    if (! -r $javaTestJar) {
        print("Unable to read the file $javaTestJar\n");
        return 0;
    }
    print("Testing Java at $javaExec\n");
    # debug lines
    #    print("Here are the details about my JVMTests call from Install.pm: \n");
    #    print("javaExec = $javaExec\n");
    #    print("javaFlags = $javaFlags\n");
    #    print("javaTestJar = $javaTestJar\n\n");
    my $exitCode = system("$javaExec $javaFlags -cp $javaTestJar com.helpsystems.enterprise.install.JVMTests -a");
    if ($exitCode > 0) {
       return 0;
   }
   # debug lines
   #print("debug fail testJava from Install.pm: \n");
   #return 0;
   return 1;
}


sub readFileHash {
    my $self = shift;
    my $filename = shift;
    my $hashmap;

	if (length($filename) == 0) {
		carp "Blank filename passed in";
		return;
	}

    if ( -r $filename ) {
        my $line;
        open (DATAFILE, $filename); 
        while (<DATAFILE>) {
            chomp($line = $_); 
            if ( length($line) == 0) {
                next;
            }
            my $c = substr($line, 0, 1);
            if ( $c eq "#") {
                # ignore it
                next;
            }
            my ($varname, $varvalue) = split('=', $line);
            if ( (! $varname) || ( length($varvalue) == 0) ) {
                #blank 
                next;
            }

            # reparse the line now that we know how long the key is
            #   This is for the goofy cases where there's an '='
            #   in the value part.
            $varvalue = substr($line, length($varname)+1);
            $self->trim($varvalue);
            $self->trim($varname);
            $hashmap->{$varname} = $varvalue;
        }
        close (DATAFILE); 
        return $hashmap;
    }
    else {
        carp("Unable to open file $filename for reading.");
        return;
    }
}

# creates a directory and changes its ownership to the automate user
#   if currently root.
sub makeDir {
    my $self = shift;
    my $name = shift;

    if ( -e $name) {
        return;
    }

    mkpath($name, 0, 0711) || print("Cannot make directory $name. $!\n");
    if (! -e $name) {
        return 0;
    }

    if ($whoami eq 'root') {
        my ($login,$pass,$uid,$gid) = getpwnam($automateUser);
        chown($uid, $gid, $name);
    }

}

# fetches a file if necessary, returns 1 if successful.
sub maybeDownload {
    my $self = shift;
    my $file = shift;
    my $listVar = shift;
    my @remainingDownloadFiles;
    my $hasMoreFiles = 0;
    if ( ref($listVar) eq 'ARRAY') {
       @remainingDownloadFiles = @$listVar;
       $hasMoreFiles = 1;
    }

    if (-e $file) {
        return 1;
    }

    my $destination = basename($file);
    
    my $ok = 0;
#    if (defined $automateScheduleDownloadURL) {
#        $ok = $self->download("$automateScheduleDownloadURL/$file", $file);
#        if ($ok == 1) {
#            return $ok;
#        }
#    }

    if ($ok == -1) {
        # don't try anymore.
        return $ok;
    }
    
    $ok = $self->download($DEFAULT_AUTOMATE_SCHEDULE_DOWNLOAD_URL . $file, $destination);
    if ($ok == 1) {
        return $ok;
    }

    if ($ok == -1) {
        # don't try anymore.
        return $ok;
    }
    
    my $uri = $DEFAULT_S3_DOWNLOAD_URL . $file;
    $ok = $self->download($uri, $destination);
    if ($ok != 1) {
        print("Unable to access the Automate Schedule download site.\n");
        if ($hasMoreFiles && defined(@remainingDownloadFiles[0])) {
            print("Please download the following files and save them in the directory\n");
        } else {
            print("Please download the following file and save it in the directory\n");
        }
        print("$startingDir\n");
        print("   " . $DEFAULT_AUTOMATE_SCHEDULE_DOWNLOAD_URL . $file . "\n");
        if ($hasMoreFiles) {
            foreach my $additionalFile (@remainingDownloadFiles) {
                print("   " . $DEFAULT_AUTOMATE_SCHEDULE_DOWNLOAD_URL . $additionalFile . "\n");
            }
        }
    }
    return $ok;
}

# downloads a file only from the S3 source, and never cache it.
sub forceDownload {
    my $self = shift;
    my $file = shift;

    my $ok = $self->download($DEFAULT_S3_DOWNLOAD_URL . $file, $file);

    return $ok;
}

# Downloads a file. Returns 1 if all is well, 0 if unsuccessful,
# and -1 if there's something seriously wrong (and an error is printed)
#
# Tries to print a message about the download in progress if it thinks
# that it's going to take a while.
sub download {

    my $self = shift;
    my $uri = shift;
    my $file = shift;

    if ( ! -w '.' ) {
        # don't bother downloading anything, because we can't write here.
        return -1;
    }

    my $protocolFailure = 0;
    my $libraryFailure;
    my $failureString;
    my $result = $userAgent->head($uri);
    if ($result->is_success) {
        my $numBytes = $result->content_length();
        if ($numBytes > 1000000) {
            $numBytes = format_bytes($numBytes);
            print("Attempting to download $uri,");
            print(" this may take a few minutes ($numBytes bytes).\n");
        }
    } else {
        $failureString = $result->as_string();
        if ( ($failureString =~ m/Protocol scheme/g) &&
            ($failureString =~ m/is not supported/g) ) {
            $protocolFailure = 1;
        } elsif ($failureString =~ m/500 Can.t locate /g) {
            $libraryFailure = $failureString;
            print("Attempting to download $uri,");
            print(" this may take a few minutes.\n");
        } else {
            # invalid URL
            return 0;
        }
    }
    
    if (-x "/usr/bin/wget" ) {
        # see if wget supports the "no-check-certificate" flag
        my $useCertificateFlag = `/usr/bin/wget --help | grep no-check-certificate | wc -l`;
        chomp($useCertificateFlag);
        $useCertificateFlag =~ s/^\s+//;
        
        my $certFlag = "";
        if ($useCertificateFlag) {
            $certFlag = "--no-check-certificate";
        }
        my $ok = system("/usr/bin/wget $certFlag -q -O $file $uri > /dev/null");
        if (($ok == 0) && (-e $file)) {
            # we're good to go.
            return 1;
        } else {
            # remove the incomplete file
            system("rm -f $file");
        }
    }

    if ($protocolFailure) {
        # don't bother.
        print("Trying to fetch $uri\n");
        print("$failureString\n");
        return 0;
    }

    $result = $userAgent->get( $uri, ':content_file' => $file);
    print ("\n");
    if ($result->is_success) {
        if (-e $file) {
            # all is well
            return 1;
        } else {
            print("Unable to write the file $file\n");
            return -1;
        }
    } else {
        # download failed
        print($result->message);
        print("\n");
        # remove the incomplete file
        system("rm -f $file");
        return 0;
    }
}


# returns 1/true if a NativeSz implementation is present for the platform
# callers ned to pass in the base dir that contains the implementations.
sub hasNativesz {

    my $self = shift;
    my $basedir = shift;
    my $platform = $self->platform();
    my $dir = "$basedir/$platform";
    if ( ! -d $dir ) {
        return 0;
    }

    my $search = new File::List($dir);
    my @files = @{ $search->find('native') };
    my $count = @files;
    if ( $count > 0 ) {
        # make sure the library and files are executable
        system("chmod -R 755 $dir");
        return 1;
    }
    return 0;
}
    

# copies a script over to another dir while replacing some text 
#   and then makes it executable.
# NOTE: The search/replace text may not contain the question mark, nor 
#       double quotes
# Returns 1 of OK, otherwise, prints an error and returns 0.
sub copyScript {
    my $self = shift;
    my $sourceScript = shift;
    my $targetScript = shift;
    my $searchText = shift;
    my $replaceText = shift;
    
    my $result = system("sed \"s?$searchText?$replaceText?g\"  $sourceScript > $targetScript");
    if ($result >  0) {
       print("Unable to copy script $sourceScript to $targetScript: $!\n");
       return 0;
    }
    $result = system("chmod 755 $targetScript");
    if ($result > 0) {
        print("Unable to set execute rights on $targetScript\n");
        return 0;
    }
    return 1;
}


# picks the first defined (and non-blank) value and returns it.
sub pickValue {
    my $self = shift;
    my $count = @_;
    while($count > 0) {
        my $value = shift;
        if (defined($value) && length($value) > 0) {
            return $value;
        }
        $count = $count -1;
    }
}

sub promptForPort {
    my $self = shift;
    my $prompt = shift;
    my $port = shift;
    my $checkIfInUse = shift;
    if (!defined($checkIfInUse)) {
        $checkIfInUse = 1;
    }

    my $input;
    while (1) {
        print("$prompt [$port] ");
        $input = <STDIN>;
        chomp($input);
        if (length($input) == 0) {
            $input = $port;
        }
        if ( (!($input =~ m/(\d){1,5}/)) || ($input < 1) || ($input > 65535) ) {
            print("Please enter a valid port number\n");
        } elsif ($checkIfInUse && $self->isPortInUse($input)) {
            print("   That port is currently in use, please select a different port.\n");
        } else {
            return $input;
        }
    }
}

sub prompt {
    my $self = shift;
    my $prompt = shift;
    my $defaultValue = shift;
    my $input;
    while (1) {
        print("$prompt");
        if (defined($defaultValue)) {
            print(" [$defaultValue] ");
        } else {
            print(": ");
        }
        $input = <STDIN>;
        chomp($input);
        if (length($input) == 0) {
            if (!defined($defaultValue)) {
                print("Please enter a value\n");
            } else {
                return $defaultValue;
            }
        } else {
            return $input;
        }
    }
}


# returns 1 for 'true' and 0 for everything else.
sub numericTrueFalse {
    my $self = shift;
    my $value = shift;
    if ('true' eq $value) {
        return 1;
    }
    return 0;
}

# given a starting port number, keep incrementing it until an open port is found.
sub findOpenPort {
    my $self = shift;
    my $port = shift;
    while ( $self->isPortInUse($port) ) {
        $port = $port -5;
    }
    return $port;
}

# returns 1 if the port appears to be in use (according to netstat)
sub isPortInUse {
    my $self = shift;
    my $port = shift;

    # netstat is deprecated on linux, and we should use ss instead
    # Try to use netstat if ss does not exist, because AIX doesnt have ss
    my $portStatCmd = "ss -an";
    my $hasSs=`command -v ss >/dev/null 2>&1`;
    
    # there is some problem with AIX getting the return code from command
    # so we are cheating here
    if (($hasSs > 0) || ($os eq 'AIX')) {
        $portStatCmd = "netstat -an";
    }

    my $in_use=`${portStatCmd} | grep LISTEN | egrep "[^0-9]${port}" | wc -l`;
    chomp($in_use);
    if ($in_use > 0) {
        return 1;
    }
    return 0;
}


sub trim {
  @_ = $_ if not @_ and defined wantarray;
  @_ = @_ if defined wantarray;
  for (@_ ? @_ : $_) { s/^\s+//, s/\s+$// }
  return wantarray ? @_ : $_[0] if defined wantarray;
}

# returns 1 if the PID in the specified file is currently running.
# returns 0 if the PID is not present in the 'ps' listing.
# returns -1 if we're unable to determine
sub checkIfProcessRunning {
    my $self = shift;
    my $pidFile = shift;

    if ( -r $pidFile ) {
        my $OLD_PID=`cat ${pidFile}`;
        if ($? != 0) {
            return -1;
        }
        chomp($OLD_PID);
        my $PROCESS_COUNT=`ps -p ${OLD_PID} | grep ${OLD_PID} | wc -l`;
        if ($? != 0) {
            return -1;
        }
        if ( ${PROCESS_COUNT} > 0 ) {
            return 1;
        } else {
            return 0;
        }
    } 
    
    return -1;
}

# waits up to 10 seconds for a process to end
sub waitForProcessEnd {
    my $self = shift;
    my $pidFile = shift;

    my $processRunning = 1;
    my $retryCount = 0;
    while ( ($processRunning == 1) && ($retryCount < 10) ) {
        $processRunning = $self->checkIfProcessRunning($pidFile);
        if ($processRunning == 1) {
            $retryCount ++;
            sleep(1);
        }
    }

    
}

# returns the GID of the automate user, or -1 if it could not be retrieved.
sub getGID {
    my $self = shift;

    if (!defined($automateUser)) {
        return -1;
    }

    if ( $os eq "Darwin" ) {
        my @info = `dscacheutil -q user -a name automate`;
        foreach my $line (@info) {
            chomp($line);
            my ($key, $value) = split /: /, $line;
            if ('gid' eq $key) {
                return $value;
            } 
        }
    } else {
        my $automateUserInfo = `grep -i ^$automateUser: /etc/passwd`;
        if ( length($automateUserInfo) == 0 ) {
            # try NIS
            $automateUserInfo = `ypcat passwd 2> /dev/null | grep -i ^$automateUser: `;
        }
        if ( length($automateUserInfo) > 0) {
            my ($user, $pw, $uid, $gid, $ignored, $dir, $shell) = split /:/, $automateUserInfo;
            return $gid;
        }
    }
    return -1;
}


1;
__END__


