#!/bin/sh

# Copyright (C) 2010 Help/Systems LLC, All Rights Reserved

# Instructs an Agent set one or more Automate Schedule Variables  
# based on constants or current values of environment variables with the same name.

logDebug="false"
MAIN_CLASS="com.helpsystems.enterprise.peer.AgentControl"

# Initialize a verbose parameter
#   - "VERBOSE=" to turn off verbose mode
#   - "VERBOSE=-v" to turn on verbose mode
VERBOSE=

# initialize empty parm flags
CFG_FILE=agent.xml
COMMAND=ROVAR
MODE=SET

# ------------------------------------------------------------------------
# Function: Log an error message                                         -
# ------------------------------------------------------------------------
logError() {
  printf "$1\n" 1>&2
}

# ------------------------------------------------------------------------
# Function: Log an info message                                         -
# ------------------------------------------------------------------------
logInfo() {
  printf "$1\n"
}

# ------------------------------------------------------------------------
# Function: Log a debug message                                          -
# ------------------------------------------------------------------------
debugMsg() {
  if test "$logDebug" = "true"
  then 
    logInfo "DEBUG: $1"
  fi
}

# ------------------------------------------------------------------------
# Function: Show Usage Text                                               -
# ------------------------------------------------------------------------
showUsageText() {
  logInfo ""
  logInfo "Usage: $0 [-h][-f config file] [-v] <Automate Schedule Variable and value>"
  logInfo "If no config file is provided, then the file agent.xml is used."
  logInfo ""
}

# ------------------------------------------------------------------------
# Function: Show Help Text                                               -
# ------------------------------------------------------------------------
showHelpText() {
  logInfo ""
  logInfo "Help for: $0"
  logInfo ""
  logInfo "SETSV - Instructs an Agent to set the value of a Automate Schedule Variable on the"
  logInfo "        Automate Schedule based on constants or current values of the"
  logInfo "        environment variable with the same name.  The Automate Schedule Variable"
  logInfo "        must exist in Automate Schedule."
  showUsageText
  logInfo "  The -h option will show command help."
  logInfo "  The -v option will run in verbose logging mode."
  logInfo ""
  logInfo "  When the Automate Schedule Variable is specified without a value, the value is "
  logInfo "  taken from the environment variable with the same name."
  logInfo ""
  logInfo "     Example: $0 VAR1"
  logInfo "     Example: $0 Var2"
  logInfo ""
  logInfo "  When the Automate Schedule Variable is specified as being set equal to a"
  logInfo "  value, the specified value is used."
  logInfo ""
  logInfo "     Example: $0 var=value"
  logInfo "     Example: $0 \"var2=two words\""
  logInfo ""
  logInfo "  When there is need to set parameter with \$ value \\\$ should be used instead."
  logInfo ""
  logInfo "  Examples:"
  logInfo "     $0 VAR1"
  logInfo "     $0 VAR1=oneword"
  logInfo "     $0 \"VAR1=two words\""
  logInfo "     $0 -f agent2.xml VAR1=test"
  logInfo "     $0 -v VAR1=test"
  logInfo "     $0 -f agent2.xml -v VAR1=test"
  logInfo "     $0 -f agent2.xml -v VAR1=\\\$test"
  logInfo ""
  logInfo "  Exit Status:"
  logInfo "           0 = Success"
  logInfo "    non-zero = Failure"
  logInfo ""
  logInfo "Copyright (C) 2010 Help/Systems LLC, All Rights Reserved"
  logInfo ""
}

# ------------------------------------------------------------------------
# Main Line Code.                                                        -
# ------------------------------------------------------------------------

if [ $# -gt 0 -a "$1" = "--help" ]; then
  showHelpText
  exit 0
fi

#Check for config file first
if [ "$1" = "-f" ]; then
   if [ -z "$2" ]; then
     echo "Error: A configuration file must follow the -f parameter"
     exit 1
   fi
fi

#-----------------------------------------------
# Process leading options.
# "Shift" them off the parm list.
#-----------------------------------------------
while getopts :vhf: opt
  do
    case $opt in
    f)   CFG_FILE=$OPTARG
         debugMsg "Using agent configuration file: $CFG_FILE"
         ;;
    h)   debugMsg "Processing -h (help) option..."
         showHelpText
         exit 0
         ;;
    v)   debugMsg "Setting on verbose mode..."
         VERBOSE=-v
         ;;
    '?') logError "$0: invalid option -$OPTARG" >&2
         showUsageText
         exit 1
         ;;
    esac
  done

shift $((OPTIND-1))

#-----------------------------------------------
# One or more variables must be specified.
#-----------------------------------------------
if [ $# -eq 0 ]; then
  logError "One or more Automate Schedule Variable names are required."
  showUsageText
  exit 1
fi

debugMsg "----------------------------------------------------------------"
debugMsg "- Verbose flag  . . . . . . : ${VERBOSE}"
debugMsg "- Using config file . . . . : ${CFG_FILE}"
debugMsg "- Sending Command . . . . . : ${COMMAND}"
debugMsg "- Mode  . . . . . . . . . . : ${MODE}"
debugMsg "----------------------------------------------------------------"
for i in "$@"
 do
debugMsg " Parameter  . . . . . . . . : $i"
 done
debugMsg "----------------------------------------------------------------"

PATH=$PATH:/usr/ucb

WORKING_DIR=`dirname $0`
cd ${WORKING_DIR}

if [ ! -f ${CFG_FILE} ]; then
    logError "The config file ${CFG_FILE} does not exist. Command failed."
    exit 1
fi

if [ ! -r ${CFG_FILE} ]; then
    echo "The config file $CFG_FILE is not readable by the current user."
    exit 1
fi

# Parameter notes:
# - The verbose parameter is not quoted because it is optional.
# - The $@ parameter will pass the SV names/values along.

./runjava.sh $MAIN_CLASS ${VERBOSE} "${CFG_FILE}" ${COMMAND} -m "${MODE}" "$@"


