"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

var _path = _interopRequireDefault(require("path"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const FILENAME_REGEX = /^[0-9a-zA-Z_\-.]+$/;

class Editor {
  constructor(bp, botId, config) {
    _defineProperty(this, "bp", void 0);

    _defineProperty(this, "_botId", void 0);

    _defineProperty(this, "_typings", void 0);

    _defineProperty(this, "_config", void 0);

    this.bp = bp;
    this._botId = botId;
    this._config = config;
  }

  async fetchFiles() {
    return {
      actionsGlobal: this._config.allowGlobal && (await this._loadFiles('/actions', 'action')),
      actionsBot: await this._loadFiles('/actions', 'action', this._botId)
    };
  }

  async _validateMetadata({
    name,
    botId,
    type
  }) {
    if (!botId || !botId.length) {
      if (!this._config.allowGlobal) {
        throw new Error(`Global files are restricted, please check your configuration`);
      }
    } else {
      if (botId !== this._botId) {
        throw new Error(`Please switch to the correct bot to change its actions.`);
      }
    }

    if (type !== 'action') {
      throw new Error('Invalid file type Only actions are allowed at the moment');
    }

    if (!FILENAME_REGEX.test(name)) {
      throw new Error('Filename has invalid characters');
    }
  }

  async saveFile(file) {
    this._validateMetadata(file);

    const {
      location,
      botId,
      content
    } = file;
    const ghost = botId ? this.bp.ghost.forBot(this._botId) : this.bp.ghost.forGlobal();
    return ghost.upsertFile('/actions', location, content);
  }

  async loadTypings() {
    if (this._typings) {
      return this._typings;
    }

    const sdkTyping = _fs.default.readFileSync(_path.default.join(__dirname, '/../botpress.d.js'), 'utf-8');

    this._typings = {
      'process.d.ts': this._buildRestrictedProcessVars(),
      'node.d.ts': this._getNodeTypings().toString(),
      'botpress.d.ts': sdkTyping.toString().replace(`'botpress/sdk'`, `sdk`)
    };
    return this._typings;
  }

  _getNodeTypings() {
    const getTypingPath = folder => _path.default.join(__dirname, `/../../${folder}/@types/node/index.d.ts`);

    if (_fs.default.existsSync(getTypingPath('node_modules'))) {
      return _fs.default.readFileSync(getTypingPath('node_modules'), 'utf-8');
    }

    return _fs.default.readFileSync(getTypingPath('node_production_modules'), 'utf-8');
  }

  async _loadFiles(rootFolder, type, botId) {
    const ghost = botId ? this.bp.ghost.forBot(botId) : this.bp.ghost.forGlobal();
    return Promise.map((await ghost.directoryListing(rootFolder, '*.js')), async filepath => {
      return {
        name: _path.default.basename(filepath),
        location: filepath,
        content: await ghost.readFileAsString(rootFolder, filepath),
        type,
        botId
      };
    });
  }

  _buildRestrictedProcessVars() {
    const exposedEnv = { ..._lodash.default.pickBy(process.env, (_value, name) => name.startsWith('EXPOSED_')),
      ..._lodash.default.pick(process.env, 'TZ', 'LANG', 'LC_ALL', 'LC_CTYPE')
    };

    const root = this._extractInfo(_lodash.default.pick(process, 'HOST', 'PORT', 'EXTERNAL_URL', 'PROXY'));

    const exposed = this._extractInfo(exposedEnv);

    return `
    declare var process: RestrictedProcess;
    interface RestrictedProcess {
      ${root.map(x => {
      return `/** Current value: ${x.value} */
${x.name}: ${x.type}
`;
    })}

      env: {
        ${exposed.map(x => {
      return `/** Current value: ${x.value} */
  ${x.name}: ${x.type}
  `;
    })}
      }
    }`;
  }

  _extractInfo(keys) {
    return Object.keys(keys).map(name => {
      return {
        name,
        value: keys[name],
        type: typeof keys[name]
      };
    });
  }

}

exports.default = Editor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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