"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _bluebird = _interopRequireDefault(require("bluebird"));

var _lodash = _interopRequireDefault(require("lodash"));

var _moment = _interopRequireDefault(require("moment"));

var _ms = _interopRequireDefault(require("ms"));

var _uuid = _interopRequireDefault(require("uuid"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class WebchatDb {
  constructor(bp) {
    this.bp = bp;

    _defineProperty(this, "knex", void 0);

    _defineProperty(this, "users", void 0);

    this.users = bp.users;
    this.knex = bp['database']; // TODO Fixme
  }

  async getUserInfo(userId) {
    const {
      result: user
    } = await this.users.getOrCreateUser('web', userId);
    const fullName = `${user.attributes['first_name']} ${user.attributes['last_name']}`;
    const avatar = user && user.attributes['picture_url'] || undefined;
    return {
      fullName,
      avatar_url: avatar
    };
  }

  async initialize() {
    return this.knex.createTableIfNotExists('web_conversations', function (table) {
      table.increments('id').primary();
      table.string('userId');
      table.string('botId');
      table.string('title');
      table.string('description');
      table.string('logo_url');
      table.timestamp('created_on');
      table.timestamp('last_heard_on'); // The last time the user interacted with the bot. Used for "recent" conversation

      table.timestamp('user_last_seen_on');
      table.timestamp('bot_last_seen_on');
    }).then(() => {
      return this.knex.createTableIfNotExists('web_messages', function (table) {
        table.string('id').primary();
        table.integer('conversationId');
        table.string('userId');
        table.string('message_type'); // @ deprecated Remove in a future release (11.9)

        table.text('message_text'); // @ deprecated Remove in a future release (11.9)

        table.jsonb('message_raw'); // @ deprecated Remove in a future release (11.9)

        table.jsonb('message_data'); // @ deprecated Remove in a future release (11.9)

        table.jsonb('payload');
        table.string('full_name');
        table.string('avatar_url');
        table.timestamp('sent_on');
      });
    }).then(() => this.knex('web_messages').columnInfo().then(info => {
      if (info.payload === undefined) {
        return this.knex.schema.alterTable('web_messages', table => {
          table.jsonb('payload');
        });
      }
    }));
  }

  async appendUserMessage(botId, userId, conversationId, payload) {
    const {
      fullName,
      avatar_url
    } = await this.getUserInfo(userId);
    const {
      type,
      text,
      raw,
      data
    } = payload;
    const convo = await this.knex('web_conversations').where({
      userId,
      id: conversationId,
      botId
    }).select('id').limit(1).then().get(0);

    if (!convo) {
      throw new Error(`Conversation "${conversationId}" not found - BP_CONV_NOT_FOUND`);
    }

    const message = {
      id: _uuid.default.v4(),
      conversationId,
      userId,
      full_name: fullName,
      avatar_url,
      message_type: type,
      message_text: text,
      message_raw: this.knex.json.set(raw),
      message_data: this.knex.json.set(data),
      payload: this.knex.json.set(payload),
      sent_on: this.knex.date.now()
    };
    return _bluebird.default.join(this.knex('web_messages').insert(message).then(), this.knex('web_conversations').where({
      id: conversationId,
      userId: userId,
      botId: botId
    }).update({
      last_heard_on: this.knex.date.now()
    }).then(), () => ({ ...message,
      sent_on: new Date(),
      message_raw: raw,
      message_data: data,
      payload: payload
    }));
  }

  async appendBotMessage(botName, botAvatar, conversationId, payload) {
    const {
      type,
      text,
      raw,
      data
    } = payload;
    const message = {
      id: _uuid.default.v4(),
      conversationId: conversationId,
      userId: undefined,
      full_name: botName,
      avatar_url: botAvatar,
      message_type: type,
      message_text: text,
      message_raw: this.knex.json.set(raw),
      message_data: this.knex.json.set(data),
      payload: this.knex.json.set(payload),
      sent_on: this.knex.date.now()
    };
    await this.knex('web_messages').insert(message).then();
    return Object.assign(message, {
      sent_on: new Date(),
      message_raw: this.knex.json.get(message.message_raw),
      message_data: this.knex.json.get(message.message_data),
      payload: this.knex.json.get(message.payload)
    });
  }

  async createConversation(botId, userId, {
    originatesFromUserMessage = false
  } = {}) {
    const uid = Math.random().toString().substr(2, 6);
    const title = `Conversation ${uid}`;
    await this.knex('web_conversations').insert({
      botId,
      userId,
      created_on: this.knex.date.now(),
      last_heard_on: originatesFromUserMessage ? this.knex.date.now() : undefined,
      title
    }).then();
    const conversation = await this.knex('web_conversations').where({
      title,
      userId,
      botId
    }).select('id').then().get(0);
    return conversation && conversation.id;
  }

  async getOrCreateRecentConversation(botId, userId, {
    originatesFromUserMessage = false
  } = {}) {
    // TODO: Lifetime config by bot
    const config = await this.bp.config.getModuleConfigForBot('channel-web', botId);
    const recentCondition = this.knex.date.isAfter('last_heard_on', (0, _moment.default)().subtract((0, _ms.default)(config.recentConversationLifetime), 'ms').toDate());
    const conversation = await this.knex('web_conversations').select('id').whereNotNull('last_heard_on').andWhere({
      userId,
      botId
    }).andWhere(recentCondition).orderBy('last_heard_on', 'desc').limit(1).then().get(0);
    return conversation ? conversation.id : this.createConversation(botId, userId, {
      originatesFromUserMessage
    });
  }

  async listConversations(userId, botId) {
    const conversations = await this.knex('web_conversations').select('id').where({
      userId,
      botId
    }).orderBy('last_heard_on', 'desc').limit(100).then();
    const conversationIds = conversations.map(c => c.id);
    let lastMessages = this.knex.from('web_messages').distinct(this.knex.raw('ON ("conversationId") *')).orderBy('conversationId').orderBy('sent_on', 'desc');

    if (this.knex.isLite) {
      const lastMessagesDate = this.knex('web_messages').whereIn('conversationId', conversationIds).groupBy('conversationId').select(this.knex.raw('max(sent_on) as date'));
      lastMessages = this.knex.from('web_messages').select('*').whereIn('sent_on', lastMessagesDate);
    }

    return this.knex.from(function () {
      this.from('web_conversations').where({
        userId,
        botId
      }).as('wc');
    }).leftJoin(lastMessages.as('wm'), 'wm.conversationId', 'wc.id').orderBy('wm.sent_on', 'desc').select('wc.id', 'wc.title', 'wc.description', 'wc.logo_url', 'wc.created_on', 'wc.last_heard_on', 'wm.message_type', 'wm.message_text', this.knex.raw('wm.full_name as message_author'), this.knex.raw('wm.avatar_url as message_author_avatar'), this.knex.raw('wm.sent_on as message_sent_on'));
  }

  async getConversation(userId, conversationId, botId) {
    const config = await this.bp.config.getModuleConfigForBot('channel-web', botId);
    const condition = {
      userId,
      botId
    };

    if (conversationId && conversationId !== 'null') {
      condition.id = conversationId;
    }

    const conversation = await this.knex('web_conversations').where(condition).then().get(0);

    if (!conversation) {
      return undefined;
    }

    const messages = await this.getConversationMessages(conversationId, config.maxMessagesHistory);
    messages.forEach(m => {
      return Object.assign(m, {
        message_raw: this.knex.json.get(m.message_raw),
        message_data: this.knex.json.get(m.message_data),
        payload: this.knex.json.get(m.payload)
      });
    });
    return Object.assign({}, conversation, {
      messages: _lodash.default.orderBy(messages, ['sent_on'], ['asc'])
    });
  }

  async getConversationMessages(conversationId, limit, fromId) {
    let query = this.knex('web_messages').where({
      conversationId: conversationId
    });

    if (fromId) {
      query = query.andWhere('id', '<', fromId);
    }

    return query.whereNot({
      message_type: 'visit'
    }).orderBy('sent_on', 'desc').limit(limit);
  }

}

exports.default = WebchatDb;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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