"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.qnaItemData = void 0;

var _axios = _interopRequireDefault(require("axios"));

var _lodash = _interopRequireDefault(require("lodash"));

var _ms = _interopRequireDefault(require("ms"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

// Handles QnA Maker API downcasing all key-values in metadata
const markUpperCase = str => str.replace(/([A-Z])/g, 'a-a-a-a-a$1a-a-a-a-a');

const restoreUpperCase = str => str.replace(/a-a-a-a-a([a-zA-Z])a-a-a-a-a/g, (_, c) => c.toUpperCase());

const keysToRestore = {
  redirectflow: 'redirectFlow',
  redirectnode: 'redirectNode'
};

const qnaItemData = ({
  questions,
  answer,
  metadata
}) => ({
  questions,
  answer,
  ..._lodash.default.fromPairs(metadata.map(({
    name,
    value
  }) => [keysToRestore[name] || name, restoreUpperCase(value)])),
  enabled: (metadata.find(({
    name
  }) => name === 'enabled') || {}).value === 'true'
});

exports.qnaItemData = qnaItemData;

const prepareMeta = data => _lodash.default.chain(data).pick(['enabled', 'action', 'redirectFlow', 'redirectNode', 'category']).toPairs().map(([name, value]) => ({
  name,
  value: _lodash.default.isString(value) ? markUpperCase(value) : value
})).filter(({
  value
}) => !_lodash.default.isUndefined(value) && value !== '').value();

const getFieldFromMetadata = (metadata, field) => metadata.find(({
  name
}) => name === field);

class MicrosoftQnaMakerStorage {
  constructor(bp, config) {
    _defineProperty(this, "bp", void 0);

    _defineProperty(this, "client", void 0);

    _defineProperty(this, "knowledgebase", void 0);

    _defineProperty(this, "endpointKey", void 0);

    _defineProperty(this, "knowledgebaseName", void 0);

    _defineProperty(this, "questions", void 0);

    _defineProperty(this, "categories", void 0);

    _defineProperty(this, "publish", () => this.client.post(`/knowledgebases/${this.knowledgebase.id}`));

    _defineProperty(this, "patchKb", params => this.client.patch(`/knowledgebases/${this.knowledgebase.id}`, params));

    _defineProperty(this, "invalidateCache", () => this.questions = undefined);

    _defineProperty(this, "waitForOperationToFinish", async (operationId, description) => {
      await Promise.delay(200);

      while (true) {
        const {
          data,
          headers: {
            'retry-after': timeout
          }
        } = await this.client.get(`/operations/${operationId}`);
        this.bp.logger.info(`[QNA] QnA Maker ${description} #${operationId} ${data.operationState}`);

        if (data.operationState === 'Failed') {
          this.bp.logger.error(data.errorResponse.error);
        }

        if (!['Running', 'NotStarted'].includes(data.operationState)) {
          return;
        }

        this.bp.logger.info(`[QNA] Waiting 3s for ${data.operationState} QnA Maker's #${operationId} operation to finish...`);
        await Promise.delay((0, _ms.default)('3s'));
      }
    });

    this.bp = bp;
    const baseURL = 'https://westus.api.cognitive.microsoft.com/qnamaker/v4.0';
    const headers = {
      'Ocp-Apim-Subscription-Key': config.qnaMakerApiKey
    };
    Object.assign(this, {
      client: _axios.default.create({
        baseURL,
        headers
      }),
      knowledgebaseName: config.qnaMakerKnowledgebase
    });

    if (config.qnaCategories && config.qnaCategories.length > 0) {
      this.categories = config.qnaCategories.split(',');
    }
  }

  async initialize() {
    const isBpKnowledgbase = ({
      name
    }) => name === this.knowledgebaseName;

    const {
      data: {
        knowledgebases: initialKnowledgebases
      }
    } = await this.client.get('/knowledgebases/');
    const existingKb = initialKnowledgebases.find(isBpKnowledgbase);

    if (existingKb) {
      this.knowledgebase = existingKb;
    } else {
      const {
        data: {
          operationId
        }
      } = await this.client.post('/knowledgebases/create', {
        name: this.knowledgebaseName
      });
      await this.waitForOperationToFinish(operationId, 'creating KB');
      const {
        data: {
          knowledgebases
        }
      } = await this.client.get('/knowledgebases/');
      this.knowledgebase = knowledgebases.find(isBpKnowledgbase);
    }

    this.endpointKey = (await this.client.get('/endpointkeys')).data.primaryEndpointKey;
  }

  async update(data, id) {
    const prevData = await this.getQuestion(id);

    const questionsChanged = _lodash.default.isEqual(data.questions, prevData.questions);

    const questionsToAdd = _lodash.default.difference(data.questions, prevData.questions);

    const questionsToDelete = _lodash.default.difference(prevData.questions, data.questions);

    const {
      data: {
        operationId
      }
    } = await this.patchKb({
      update: {
        qnaList: [{
          id,
          answer: data.answer,
          ...(questionsChanged ? {} : {
            questions: {
              add: questionsToAdd,
              delete: questionsToDelete
            }
          }),
          metadata: {
            delete: prevData['metadata'],
            add: prepareMeta(data)
          }
        }]
      }
    });
    await this.waitForOperationToFinish(operationId, 'Updating QNA Items');
    this.invalidateCache();
    await this.publish();
    return id;
  }

  async insert(qna) {
    const qnas = _lodash.default.isArray(qna) ? qna : [qna];
    const {
      data: {
        operationId
      }
    } = await this.patchKb({
      add: {
        qnaList: qnas.map(qna => ({
          answer: qna.answer,
          questions: qna.questions.reverse(),
          // To be able to prepend questions
          metadata: prepareMeta(qna)
        }))
      }
    });
    await this.waitForOperationToFinish(operationId, 'inserting qnaItems');
    this.invalidateCache();
    await this.publish(); // TODO: should return ids (for consistency)
  }

  async fetchAllQuestions() {
    if (!this.questions) {
      const {
        data: {
          qnaDocuments
        }
      } = await this.client.get(`/knowledgebases/${this.knowledgebase.id}/test/qna/`); // Showing latest items first

      this.questions = qnaDocuments.reverse().map(doc => ({ ...doc,
        questions: doc.questions.reverse()
      }));
    }

    return this.questions;
  }

  async fetchQuestionsWithPaging(paging) {
    let questions = await this.fetchAllQuestions();

    if (paging && paging.start && paging.count) {
      questions = questions.slice(paging.start, paging.start + paging.count);
    }

    return questions.map(qna => ({
      id: qna.id,
      data: qnaItemData(qna)
    }));
  }

  async filterByCategoryAndQuestion({
    question,
    categories
  }) {
    const allQuestions = await this.fetchAllQuestions();
    const filteredQuestions = allQuestions.filter(({
      questions,
      metadata
    }) => {
      const category = getFieldFromMetadata(metadata, 'category');
      const isRightId = questions.join('\n').toLowerCase().indexOf(question.toLowerCase()) !== -1;

      if (!categories.length) {
        return isRightId;
      }

      if (!question) {
        return category && categories.indexOf(category.value) !== -1;
      }

      return isRightId && category && categories.indexOf(category.value) !== -1;
    });
    const questions = filteredQuestions.reverse().map(qna => ({
      id: qna.id,
      data: qnaItemData(qna)
    }));
    return questions;
  }

  async getQuestions({
    question = '',
    categories = []
  }, {
    limit = 50,
    offset = 0
  }) {
    let items = [];
    let count = 0;

    if (!(question || categories.length)) {
      items = await this.fetchQuestionsWithPaging({
        start: offset ? parseInt(offset) : undefined,
        count: limit ? parseInt(limit) : undefined
      });
      count = await this.count();
    } else {
      const tmpQuestions = await this.filterByCategoryAndQuestion({
        question,
        categories
      });
      items = tmpQuestions.slice(offset, offset + limit);
      count = tmpQuestions.length;
    }

    return {
      items,
      count
    };
  }

  async getQuestion(id) {
    const question = (await this.fetchAllQuestions()).find(({
      id: qnaId
    }) => qnaId == id);
    return question && qnaItemData(question);
  }

  async count() {
    const questions = await this.fetchAllQuestions();
    return questions.length;
  }

  async answersOn(question, category = undefined) {
    const metadataFilters = category ? [{
      name: 'category',
      value: category
    }] : [];
    const resp = await _axios.default.post(`/qnamaker/knowledgebases/${this.knowledgebase.id}/generateAnswer`, {
      question,
      top: 10,
      strictFilters: [{
        name: 'enabled',
        value: true
      }, ...metadataFilters]
    }, {
      baseURL: this.knowledgebase.hostName,
      headers: {
        Authorization: `EndpointKey ${this.endpointKey}`
      }
    });
    const {
      data: {
        answers
      }
    } = resp;
    return _lodash.default.orderBy(answers, ['score'], ['desc']).map(answer => ({ ..._lodash.default.pick(answer, ['questions', 'answer', 'id', 'metadata', 'enabled']),
      confidence: answer.score / 100,
      ...qnaItemData(answer)
    }));
  }

  async delete(id, statusCb) {
    const ids = _lodash.default.isArray(id) ? id : [id];

    if (ids.length === 0) {
      return;
    }

    const maxQuestionsToDeletePerRequest = 300;
    await Promise.each(_lodash.default.chunk(ids, maxQuestionsToDeletePerRequest), async (idsChunk, i) => {
      const {
        data: {
          operationId
        }
      } = await this.client.patch(`/knowledgebases/${this.knowledgebase.id}`, {
        delete: {
          ids: idsChunk
        }
      });
      await this.waitForOperationToFinish(operationId, 'deleting qnaItems');
      statusCb && statusCb(Math.min((i + 1) * maxQuestionsToDeletePerRequest, ids.length));
    });
    this.invalidateCache();
    await this.publish();
  }

  getCategories() {
    return this.categories;
  }

  hasCategories() {
    return this.categories && this.categories.length > 0;
  }

}

exports.default = MicrosoftQnaMakerStorage;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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