"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _iconvLite = _interopRequireDefault(require("iconv-lite"));

var _joi = require("joi");

var _json2csv = require("json2csv");

var _moment = _interopRequireDefault(require("moment"));

var _multer = _interopRequireDefault(require("multer"));

var _nanoid = _interopRequireDefault(require("nanoid"));

var _yn = _interopRequireDefault(require("yn"));

var _transfer = require("./transfer");

var _validation = require("./validation");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var _default = async (bp, botScopedStorage) => {
  const csvUploadStatuses = {};
  const router = bp.http.createRouterForBot('qna');
  router.get('/questions', async (req, res) => {
    try {
      const {
        query: {
          question = '',
          categories = [],
          limit,
          offset
        }
      } = req;
      const storage = botScopedStorage.get(req.params.botId);
      const items = await storage.getQuestions({
        question,
        categories
      }, {
        limit,
        offset
      });
      res.send({ ...items
      });
    } catch (e) {
      bp.logger.attachError(e).error('Error listing questions');
      res.status(500).send(e.message || 'Error');
    }
  });
  router.post('/questions', async (req, res, next) => {
    try {
      const qnaEntry = await (0, _joi.validate)(req.body, _validation.QnaDefSchema);
      const storage = botScopedStorage.get(req.params.botId);
      const id = await storage.insert(qnaEntry);
      res.send(id);
    } catch (e) {
      next(new Error(e.message));
    }
  });
  router.get('/questions/:id', async (req, res) => {
    try {
      const storage = botScopedStorage.get(req.params.botId);
      const question = await storage.getQuestion(req.params.id);
      res.send(question);
    } catch (e) {
      sendToastError('Fetch', e.message);
    }
  });
  router.put('/questions/:id', async (req, res, next) => {
    const {
      query: {
        limit,
        offset,
        question,
        categories
      }
    } = req;

    try {
      const qnaEntry = await (0, _joi.validate)(req.body, _validation.QnaDefSchema);
      const storage = botScopedStorage.get(req.params.botId);
      await storage.update(qnaEntry, req.params.id);
      const questions = await storage.getQuestions({
        question,
        categories
      }, {
        limit,
        offset
      });
      res.send(questions);
    } catch (e) {
      next(new Error(e.message));
    }
  });
  router.delete('/questions/:id', async (req, res) => {
    const {
      query: {
        limit,
        offset,
        question,
        categories
      }
    } = req;

    try {
      const storage = botScopedStorage.get(req.params.botId);
      await storage.delete(req.params.id, undefined);
      const questionsData = await storage.getQuestions({
        question,
        categories
      }, {
        limit,
        offset
      });
      res.send(questionsData);
    } catch (e) {
      bp.logger.attachError(e).error('Could not delete QnA #' + req.params.id);
      res.status(500).send(e.message || 'Error');
      sendToastError('Delete', e.message);
    }
  });
  router.get('/categories', async (req, res) => {
    const storage = botScopedStorage.get(req.params.botId);
    const categories = await storage.getCategories();
    res.send({
      categories
    });
  });
  router.get('/export/:format', async (req, res) => {
    const storage = botScopedStorage.get(req.params.botId);
    const config = await bp.config.getModuleConfigForBot('qna', req.params.botId);
    const data = await (0, _transfer.prepareExport)(storage, {
      flat: true
    });

    if (req.params.format === 'csv') {
      res.setHeader('Content-Type', 'text/csv');
      res.setHeader('Content-disposition', `attachment; filename=qna_${(0, _moment.default)().format('DD-MM-YYYY')}.csv`);
      const categoryWrapper = storage.hasCategories() ? ['category'] : [];
      const parseOptions = {
        fields: ['question', 'action', 'answer', 'answer2', ...categoryWrapper],
        header: true
      };
      const json2csvParser = new _json2csv.Parser(parseOptions);
      res.end(_iconvLite.default.encode(json2csvParser.parse(data), config.exportCsvEncoding));
    } else {
      res.setHeader('Content-Type', 'application/json');
      res.setHeader('Content-disposition', `attachment; filename=qna_${(0, _moment.default)().format('DD-MM-YYYY')}.json`);
      res.end(JSON.stringify(data));
    }
  });
  const upload = (0, _multer.default)();
  router.post('/import/csv', upload.single('csv'), async (req, res) => {
    const storage = botScopedStorage.get(req.params.botId);
    const config = await bp.config.getModuleConfigForBot('qna', req.params.botId);
    const uploadStatusId = (0, _nanoid.default)();
    res.end(uploadStatusId);
    updateUploadStatus(uploadStatusId, 'Deleting existing questions');

    if ((0, _yn.default)(req.body.isReplace)) {
      const questions = await storage.fetchAllQuestions();

      const statusCb = processedCount => updateUploadStatus(uploadStatusId, `Deleted ${processedCount}/${questions.length} questions`);

      await storage.delete(questions.map(({
        id
      }) => id), statusCb);
    }

    try {
      const questions = _iconvLite.default.decode(req.file.buffer, config.exportCsvEncoding);

      const params = {
        storage,
        config,
        format: 'csv',
        statusCallback: updateUploadStatus,
        uploadStatusId
      };
      await (0, _transfer.importQuestions)(questions, params);
      updateUploadStatus(uploadStatusId, 'Completed');
    } catch (e) {
      bp.logger.attachError(e).error('CSV Import Failure');
      updateUploadStatus(uploadStatusId, `Error: ${e.message}`);
    }
  });
  router.get('/csv-upload-status/:uploadStatusId', async (req, res) => {
    res.end(csvUploadStatuses[req.params.uploadStatusId]);
  });

  const sendToastError = (action, error) => {
    bp.realtime.sendPayload(bp.RealTimePayload.forAdmins('toast.qna-save', {
      text: `QnA ${action} Error: ${error}`,
      type: 'error'
    }));
  };

  const updateUploadStatus = (uploadStatusId, status) => {
    if (!uploadStatusId) {
      return;
    }

    csvUploadStatuses[uploadStatusId] = status;
  };
};

exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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