"use strict";

const base = require('./_base');

const path = require('path');

const url = require('url');

const {
  tail
} = _;

function render(data) {
  const events = [];

  if (data.typing) {
    events.push({
      type: 'typing',
      value: data.typing
    });
  }

  return [...events, {
    type: 'file',
    url: url.resolve(data.BOT_URL, data.image)
  }];
}

function renderMessenger(data) {
  return [{
    type: 'typing',
    value: data.typing
  }, {
    attachment: {
      type: 'template',
      payload: {
        template_type: 'generic',
        elements: [{
          title: data.title,
          image_url: url.resolve(data.BOT_URL, data.image)
        }]
      }
    }
  }];
}

function renderElement(data, channel) {
  if (channel === 'web' || channel === 'api') {
    return render(data);
  } else if (channel === 'messenger') {
    return renderMessenger(data);
  }

  return []; // TODO Handle channel not supported
}

module.exports = {
  id: 'builtin_image',
  group: 'Built-in Messages',
  title: 'Image',
  jsonSchema: {
    description: 'A message showing an image with an optional title',
    type: 'object',
    required: ['image'],
    properties: {
      image: {
        type: 'string',
        $subtype: 'media',
        $filter: '.jpg, .png, .jpeg, .gif, .bmp, .tif, .tiff|image/*',
        title: 'Image'
      },
      title: {
        type: 'string',
        description: 'Some platforms require to name the images.',
        title: 'Title (optional)'
      },
      ...base.typingIndicators
    }
  },
  uiSchema: {
    title: {
      'ui:field': 'i18n_field'
    }
  },
  computePreviewText: formData => {
    if (!formData.image) {
      return;
    }

    let fileName = path.basename(formData.image);

    if (fileName.includes('-')) {
      fileName = tail(fileName.split('-')).join('-');
    }

    const title = formData.title ? ' | ' + formData.title : '';
    return `Image (${fileName}) ${title}`;
  },
  renderElement: renderElement
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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