"use strict";

const base = require('./_base');

const Card = require('./card');

const url = require('url');

function render(data) {
  const events = [];

  if (data.typing) {
    events.push({
      type: 'typing',
      value: data.typing
    });
  }

  return [...events, {
    text: ' ',
    type: 'carousel',
    elements: data.items.map(card => ({
      title: card.title,
      picture: card.image ? url.resolve(data.BOT_URL, card.image) : null,
      subtitle: card.subtitle,
      buttons: (card.actions || []).map(a => {
        if (a.action === 'Say something') {
          return {
            type: 'say_something',
            title: a.title,
            text: a.text
          };
        } else if (a.action === 'Open URL') {
          return {
            type: 'open_url',
            title: a.title,
            url: a.url && a.url.replace('BOT_URL', data.BOT_URL)
          };
        } else if (a.action === 'Postback') {
          return {
            type: 'postback',
            title: a.title,
            payload: a.payload
          };
        } else {
          throw new Error(`Webchat carousel does not support "${a.action}" action-buttons at the moment`);
        }
      })
    }))
  }];
}

function renderMessenger(data) {
  const renderElements = data => {
    return data.items.map(card => ({
      title: card.title,
      image_url: card.image ? url.resolve(data.BOT_URL, card.image) : null,
      subtitle: card.subtitle,
      buttons: (card.actions || []).map(a => {
        if (a.action === 'Say something') {
          throw new Error('Channel-Messenger carousel does not support "Say something" action-buttons at the moment');
        } else if (a.action === 'Open URL') {
          return {
            type: 'web_url',
            url: a.url,
            title: a.title
          };
        } else {
          throw new Error(`Channel-Messenger carousel does not support "${a.action}" action-buttons at the moment`);
        }
      })
    }));
  };

  return [{
    type: 'typing',
    value: data.typing
  }, {
    attachment: {
      type: 'template',
      payload: {
        template_type: 'generic',
        elements: renderElements(data)
      }
    }
  }];
}

function renderElement(data, channel) {
  if (channel === 'web' || channel === 'api' || channel === 'telegram') {
    return render(data);
  } else if (channel === 'messenger') {
    return renderMessenger(data);
  }

  return []; // TODO Handle channel not supported
}

module.exports = {
  id: 'builtin_carousel',
  group: 'Built-in Messages',
  title: 'Carousel',
  jsonSchema: {
    description: 'A carousel is an array of cards',
    type: 'object',
    required: ['items'],
    properties: {
      items: {
        type: 'array',
        title: 'Carousel Cards',
        items: Card.jsonSchema
      },
      ...base.typingIndicators
    }
  },
  computePreviewText: formData => formData.items && `Carousel: (${formData.items.length}) ${formData.items[0].title}`,
  renderElement: renderElement
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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