"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findMostConfidentIntentMeanStd = findMostConfidentIntentMeanStd;
exports.createIntentMatcher = exports.NoneIntent = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _patternsUtils = require("../../tools/patterns-utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const NoneIntent = {
  confidence: 1.0,
  name: 'none',
  context: 'global'
  /**
   * Finds the most confident intent, either by the intent being above a fixed threshold, or else if an intent is more than {@param zThresh} standard deviation (outlier method) from the mean.
   * NOTE: If you ever need this in another context, we could move this into tools and replace the "intent" concept for a more generic "prediction"
   * @param intents
   * @param fixedThreshold
   * @param zThresh number of standard deviation between 2 furthest from the mean
   */

};
exports.NoneIntent = NoneIntent;

function findMostConfidentIntentMeanStd(intents, fixedThreshold, zThresh = 1.15) {
  if (!intents.length) {
    return NoneIntent;
  }

  const best = intents.find(x => x.confidence >= fixedThreshold);

  if (best) {
    return best;
  }

  if (intents.length < 3) {
    return NoneIntent;
  }

  const mean = _lodash.default.meanBy(intents, 'confidence');

  const stdDeviation = Math.sqrt(intents.reduce((a, c) => a + Math.pow(c.confidence - mean, 2), 0) / (intents.length - 1));
  const zintents = intents.map(intent => ({
    intent,
    z: _lodash.default.round((intent.confidence - mean) / stdDeviation, 2)
  })).sort((a, b) => a.z > b.z ? -1 : 1);
  return zintents[0].z - zintents[1].z >= zThresh ? zintents[0].intent : NoneIntent;
}

const createIntentMatcher = intentName => {
  return pattern => {
    const matcher = new RegExp(`^${(0, _patternsUtils.escapeRegex)(pattern)}$`, 'i');
    return matcher.test(intentName);
  };
};

exports.createIntentMatcher = createIntentMatcher;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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