#!/usr/bin/perl

use strict;
# there isn't a _reliable_ way to dynamically include a module and have
#   it successfully resolve its dependencies
use lib 'include';

use AutomateSchedule::Install;
use File::List;
use Cwd;

umask 022;

my $startingDir = cwd();
my $fileDir = 'files-agent';
my $installer = AutomateSchedule::Install->new($fileDir) or die "Unable to load Automate Schedule Installer module.";
my $os = $installer->os();
my $rawArch = $installer->rawArch();
my $arch = $installer->arch();
my $platform = $installer->platform();
my $bitWidth = $installer->bitWidth();
my $osversion = $installer->osVersion();
my $useJavaVersion = $installer->useJavaVersion();
my $jreVersion;
my $whoami = $installer->whoami();
my $startAgent = 1;
my $execAgent = 0;
$ENV{'LD_LIBRARY_PATH'} = $ENV{'LD_LIBRARY_PATH'} . ":$fileDir/bin/$platform";

my $skybotServerUrl;
my $agentInstallConfigName = 'unix_default';
my $quietInstall = 0;
my $agentDirOverride;
my $manualInstall = 1;
my $argc = @ARGV;
if ( $argc > 0 ) {
    $manualInstall = 0;

    # parse the command line args
    while (1) {
        my $argv = shift;
        if (!defined($argv)) {
            last;
        }

        my $arg = lc $argv;
        if ($arg eq "-q") {
            $quietInstall = 1;
        } elsif ( $arg eq "-d") {
            $agentDirOverride = shift;
            if (!defined($agentDirOverride)) {
                fail("Please specify the agent directory");
            }
            $manualInstall = 1;
            next;
        } elsif ( $arg eq "-nostart") {
            $startAgent = 0;
        } elsif ( $arg eq "-useexec") {
            $execAgent = 1;
        } elsif ( ($arg =~ m/^http:\/\//) || ($arg =~ m/^https:\/\//) 
            || ($arg =~ m/^local:\/\//) )  {
            # looks like a url
            $skybotServerUrl = $arg;
        } else {
            $agentInstallConfigName = $argv;
        }
    }

    if ( (!defined($skybotServerUrl)) && ($manualInstall == 0) ) {
        print("Error: Invalid Automate Schedule Server address: $ARGV[0]\n");
        showHelpAndExit();
    }
}

if ( defined($skybotServerUrl) && defined($agentDirOverride) ) {
    fail("Error: The -d flag cannot be used with an Agent Install Configuration");
}

if ($quietInstall == 0) {
    print("\nWelcome to Automate Schedule Agent, version " . $installer->buildInfo() . "\n");
    print("The current user is $whoami, this system is $platform $osversion, $bitWidth bits\n\n");
} else {	
    my $now_string = `date "+%Y-%m-%d %H:%m:%S"`;
    print("Starting installation of the Automate Schedule Agent, version " . $installer->buildInfo()." : ".$now_string. "\n");
}

$installer->hasNativesz("$fileDir/bin/") || 
    fail("Sorry, but this platform ($platform) is not supported.");

my $autoConfig;
my $autoCreateSkybotUser = -1;
my $autoInstallLocalJava = -1;
my $autoInstallDir;
my $autoListenPort;
my $autoJavaHome;
my $autoSkybotServerAddress;
my $autoSkybotServerPort;
if (defined($skybotServerUrl)) {
    # download the config
    my ($protocol, $zilch, $addressAndPort, $ignored) = split /\//, $skybotServerUrl;
    $skybotServerUrl = $protocol . '//' . $addressAndPort;

    my $cleanupConfigFile = 0;
    if ($protocol ne 'local:') { 
        my $result = $installer->download("$skybotServerUrl/automate-schedule/agent_install_configurations/$agentInstallConfigName/get", $agentInstallConfigName);
        if ($result != 1) {
            fail("Unable to retrieve the Agent Install Configuration from the requested server.\nPlease check the server address, or run the installer without any parameters\nand manually configure it.");
        }
        $installer->setAutomateScheduleDownloadURL("$skybotServerUrl/automate-schedule/download");
        $cleanupConfigFile = 1;
    }

    if ( ! -r $agentInstallConfigName) {
        fail("Unable to read file $agentInstallConfigName");
    }
    $autoConfig = $installer->readFileHash($agentInstallConfigName);
    if ($cleanupConfigFile) {
        system("rm -f $agentInstallConfigName");
    }

    my $fetchError = $autoConfig->{'Error'};
    if (length($fetchError) > 0) {
        fail("Error fetching the information from the server.\n$fetchError\n");
    }

    my $autoJavaHome = $autoConfig->{'JavaHome'};
    if (length($autoJavaHome) > 0) {
        $ENV{'JAVA_HOME'} = $autoJavaHome;
    }
    if ('true' eq $autoConfig->{'CreateSkybotUser'}) {
        $autoCreateSkybotUser = 1;
    } else {
        $autoCreateSkybotUser = 0;
    }

    if ('true' eq $autoConfig->{'InstallLocalJava'}) {
        $autoInstallLocalJava = 1;
    } else {
        $autoInstallLocalJava = 0;
    }
    
    $autoInstallDir = $autoConfig->{'InstallDirectory'};
    $autoListenPort = $autoConfig->{'LocalListenPort'};
    if ($installer->isPortInUse($autoListenPort)) {
        fail("The Agent Install Configuration indicates that port $autoListenPort should be used\n   for local connections to the Agent, however that port is currently in use.");
    }
    $autoSkybotServerPort = $autoConfig->{'ServerListenPort'};
    if ($protocol eq 'local:') {
        $autoSkybotServerAddress = `hostname`;
        chomp($autoSkybotServerAddress);
    } else {
        my ($address, $port) = split /:/, $addressAndPort;
        $autoSkybotServerAddress = $address;
    }
}

my $automateUser;
my $automateScheduleDir;
my $oldSkybotDir;
my $createAutomateUser = 0;
my $rebrandInSkybot = 0;

#### For testing.
# print("whoami = $whoami\n");

if (($whoami eq 'root') || ($whoami =~ m/automate/i)) {
    # See if there's a automate user defined
    $automateUser = $installer->automateUser();
    $automateScheduleDir = $installer->automateScheduleDir();

    if (!defined($automateUser) && ($whoami eq 'root') ) {
        if ( $autoCreateSkybotUser == 0 ) {
            fail("The Automate Schedule files need to be owned by a non-root user,\n" .
                 "   and the Agent Installer Configuration '$agentInstallConfigName'\n" .
                 "   has instructed this installer not to create a local 'automate' user.\n" .
                 "Please run this installer as a non-root user, or manually add the 'automate' user.");
        }
        print("* The Automate Schedule files need to be owned by a non-root user,\n");
        print("  a local 'automate' user will be created.\n");
        $automateUser = 'automate';
        $installer->setAutomateUser($automateUser);
        $automateScheduleDir = '/opt/automate-schedule';
        $oldSkybotDir = '/opt/skybot';
        $createAutomateUser = 1;
        $installer->setAutomateScheduleDir($automateScheduleDir);
        $installer->setUserHome($automateScheduleDir);
    }
    
} elsif ($whoami =~ m/skybot/i) {
    # This is a rebrand as skybot so we cannot move to automate-schedule.
    # Treat this as a normal update and leave the skybot user.
    $rebrandInSkybot = 1;
    $automateUser = $whoami;
    $installer->setAutomateUser($automateUser);
    $automateScheduleDir = $installer->userHome() . "/skybot";
    $installer->setAutomateScheduleDir($automateScheduleDir);
} else {
    # install it into a 'automate-schedule' dir in their home dir
    $automateUser = $whoami;
    $automateScheduleDir = $installer->userHome() . "/automate-schedule";
    $oldSkybotDir = $installer->userHome() . "/skybot";
    $installer->setAutomateScheduleDir($automateScheduleDir);
}

#### For testing.
# print("automateScheduleDir = $automateScheduleDir\n");
# print("autoInstallDir = $autoInstallDir\n");
# print("agentDirOverride = $agentDirOverride\n");

##### JEL added for rebranding.
my $oldSkybotAgentDir = "$oldSkybotDir/agent";
my $rebrand = 0;
my $skybotGID;

if ($rebrandInSkybot == 0) {
    # print("checking for $oldSkybotAgentDir/lib/schent-mp-module.jar\n");
    if ( -e "$oldSkybotAgentDir/lib/schent-mp-module.jar") {
        $rebrand = 1;
        print("* Skybot will be rebranded to Automate Schedule.\n");
    }
}

if (defined($autoInstallDir)) {
    $automateScheduleDir = $autoInstallDir;
    $installer->setAutomateScheduleDir($automateScheduleDir);
    $installer->setUserHome($automateScheduleDir);
}

my $skybotAgentDir = "$automateScheduleDir/agent";

# Only if not a rebrand update.  Otherwise rebrand tries mv to the wrong folder.
if ($rebrand == 0 && defined($agentDirOverride)) {
    $automateScheduleDir = $agentDirOverride;
    $installer->setAutomateScheduleDir($automateScheduleDir);
    $installer->setUserHome($automateScheduleDir);
    $skybotAgentDir = $agentDirOverride;
}

if ($rebrand == 1) {
    checkIfOldSkybotAgentRunning();
} else {
    checkIfAgentRunning();
}

# resolve Java
my $javaInstallFile;
my $javaInstallCmd;
my $javaDir = $installer->findJava();
my $jre;

# added to prevent java 1.6 installations.
if ($useJavaVersion == 6) {
	fail("Sorry, but this platform ($platform) os ($os) and architecture ($arch) require an unsupported java version.");
}

if (defined($javaDir)) {
    print("* Using Java at $javaDir\n");
    $installer->hasNativesz("$fileDir/bin/") || 
        fail("This platform ($platform) is not supported.");

    testNativesz();
    
} else {
    if ($autoInstallLocalJava == 0) {
        fail("Error: Unable to locate a usable Java installation, and the\n" .
            "   Agent Install Configuration '$agentInstallConfigName'\n" .
            "   instructs us not to install a local copy of Java.");
    }

    if (defined($agentDirOverride)) {
        fail("Error: Unable to locate a usable Java installation. Please point the\n" .
            "   environment variable JAVA_HOME at the directory containing Java 1.7\n");
    }

    #debug testing
    #print("Install.pm :: osversion is $osversion.\n");
    #$osversion = 10.6;
    #print("Install.pm :: fake osversion is $osversion.\n");

    $jre = "jre1${useJavaVersion}";
    $jreVersion = "$useJavaVersion";
    
 

##### if ($useJavaVersion == 7) {
#####     $jre = "jre1${useJavaVersion}2";
#####     print("NOTE =======> Test override for jre17 installer: $jre\n");
##### }

    if ( ($os eq 'Linux') &&
        (($arch eq 'i686') || ($arch eq 's390x')) ) {
            $javaInstallFile = "$jre-$os-$arch.tgz";
            $javaInstallCmd = 'tar xzf _FILE_';
    }

    elsif ( ($os eq 'Linux') &&
        (($arch eq 'x86_64') || ($arch eq 'ppc64') || ($arch eq 'ppc64le')) ) {
            $javaInstallFile = "${jre}_161-$os-$arch.tgz";
            $javaInstallCmd = 'tar xzf _FILE_';
    }

    elsif ( ($os eq 'AIX') && ($rawArch eq 'powerpc') ) {
        $javaInstallFile = "$jre-$os-$arch.tar.Z";
        $javaInstallCmd = 'zcat _FILE_ | tar -xf -';
    }

    # For SunOS - the same package contains 32-bit as well as 64-bit support for the JVM
    elsif ($os eq 'SunOS') {
         if ($rawArch eq 'sparc') {
             $javaInstallFile = "$jre-$os-$rawArch.tar.Z";
             $javaInstallCmd = 'zcat _FILE_ | tar -xf -';
         } else {
             $javaInstallFile = "$jre-$os-i386.tar.bz2";
             $javaInstallCmd = 'bzip2 -cd _FILE_ | tar -xf -';
        }
    }

    elsif ($os eq 'HP-UX') {
        $javaInstallFile = "$jre-$os-$arch.tar.Z";
        $javaInstallCmd = 'zcat _FILE_ | tar -xf -';
    }

    # Mac OS JRE became distributable with 1.7 - works on 10.7 and up
    elsif ( ($os eq 'Darwin') && ($rawArch eq 'x86_64') && ($osversion >= 10.7) )  {
        $javaInstallFile = "$jre-$os-$arch.tar.Z";
        $javaInstallCmd = 'zcat _FILE_ | tar -xf -';
    }

    if (!defined($javaInstallFile)) {
        # we don't have an installer for this combination.
        print("* Unable to find a usable Java Runtime Environment (JRE).\n");
        print("  Please download and install Java version 1.7 or 1.8 and point the environment\n");
        print("      variable JAVA_HOME at the directory.\n");
        print("  Visit http://www.helpsystems.com/getjava for more info.\n");
        exit 1;
    }

    $javaDir = "$automateScheduleDir/jre";
    #debug lines
    #    print("from installAgent.pl: planning to install $jre\n\n");
    print ("* Java $jreVersion will be installed at $javaDir\n");
}

# make sure this is set correctly
$ENV{'JAVA_HOME'} = $javaDir;

my $tlPort;
my $serverAddress;
my $serverPort;

# see if we're installing or updating.
my $install = 1;
my $currentProductAgentDir = $skybotAgentDir;
if ($rebrand == 1) {
    $currentProductAgentDir = $oldSkybotAgentDir;
}
print("* Looking for $currentProductAgentDir/agent.xml\n");
if ( -e "$currentProductAgentDir/agent.xml") {
    # looks like an UPDATE
    $install = 0;
    print("* The Automate Schedule Agent will be updated at $currentProductAgentDir\n");
    #my $line = `grep ListenPort $currentProductAgentDir/agent.xml` or
    #    fail("Unable to read the existing configuration: $!");
    #chomp($line);
    #my @pieces = split /"/, $line;
    #$tlPort = $pieces[5];
} else {
    print("* The Automate Schedule Agent will be installed at $skybotAgentDir\n");
    if ( defined($skybotServerUrl) ) {
        # automatic install
        $tlPort = $autoListenPort;
        $serverAddress = $autoSkybotServerAddress;
        $serverPort = $autoSkybotServerPort;
    } else {
        $tlPort = $installer->findOpenPort(7471);
        manualConfig();
    }
}

if ($quietInstall) {
    print("Installation will proceed in 5 seconds. Press Control-C to cancel.\n");
    sleep(5);
} else {
    print("\nPress 'Enter' to continue with the installation,\n");
    print("  or press Control-C to exit.\n");
    my $input = <STDIN>;
}

## JEL Changed.  If we are not rebranding, create the product user here.
if ($rebrand == 0) {
    createProductUser();
} else {
    # If we are rebranding, the product user will be created in the sub.
    moveProductFolder();
}

# determine which SAP library we'll eventually need.
my $sapFile = "libsapjco3.so";
    
if ( $arch eq '9000' ) {
    $sapFile = "libsapjco3.sl";
}
if ( $os eq 'Darwin' ) {
    $sapFile = "libsapjco3.jnilib";
}
my $sapLibPath = "bin/$platform/$sapFile";
my @remainingDownloads = ($sapLibPath);
    

if (defined($javaInstallFile)) {
    installJava();

    testNativesz();
}


if ( ! -e $skybotAgentDir) {
    $installer->makeDir($skybotAgentDir);
    system("chmod 755 $skybotAgentDir") == 0 || fail("Unable to set permissions on $skybotAgentDir");
}

my $agentAuthDir = $skybotAgentDir . '/authorized-agents';
if ( ! -e $agentAuthDir) {
    $installer->makeDir($agentAuthDir);
    system("chmod 700 $agentAuthDir") == 0 || fail("Unable to set permissions on $agentAuthDir");
}

downloadExtraBinaries();
copyFiles();
configAgentScripts();

if ($install == 1) {
    configInfo();
} else {
## This is an update. Remove these files.
    system("rm -f \"$skybotAgentDir\"/SkytbotLicenseAgreement.html > /dev/null 2>&1");
    system("rm -f \"$skybotAgentDir\"/lib/commons-net-2.0.jar > /dev/null 2>&1");
    system("rm -f \"$skybotAgentDir\"/lib/jasperreports-4.0.0.jar > /dev/null 2>&1");
    system("rm -f \"$skybotAgentDir\"/lib/jsch-0.1.44.jar > /dev/null 2>&1");
}

# If this is a rebrand, remove these files.
if ($rebrand == 1 || $rebrandInSkybot == 1) {
    system("rm -f $skybotAgentDir/SkybotLicenseAgreement.html > /dev/null 2>&1") == 0 || fail("Unable to remove $skybotAgentDir/SkytbotLicenseAgreement.html");
    system("rm -f $skybotAgentDir/skybot-import-cron.sh > /dev/null 2>&1") == 0 || fail("Unable to remove $skybotAgentDir/skybot-import-cron.sh");
}

## JEL Added sub for creating user automate so it can be called earlier for
## a rebranding.
sub createProductUser {
	my $fixDirPermissions = 0;
	if ($createAutomateUser) {
	    if ( $os eq "Darwin" ) {
            # OSX is a little different
            if (createSkybotUserForOSX() == 0) {
                fail("Unable to create the 'automate' user.");
            }
        } else {
            system("/usr/sbin/useradd -d $automateScheduleDir automate") == 0 || fail("Unable to create 'automate' user.");
        }

        $fixDirPermissions = 1;
    }

	$skybotGID = $installer->getGID();

	# Time to start installing files to the destination dir.
	if ( ! -e $automateScheduleDir) {
    	$installer->makeDir($automateScheduleDir);
    	$fixDirPermissions = 1;
	}

	if ($fixDirPermissions) {
    	system("chmod 755 $automateScheduleDir") == 0 || die("Unable to set permissions on $automateScheduleDir\n");
	}
	
}

## JEL Added sub for moving product folder and creating product user when rebranding.
sub moveProductFolder {
    
    # If we get here OK to move files for rebranding.
    # But only if not already moved.
    if ( -e $automateScheduleDir) {
        print("* Product has already been moved to $automateScheduleDir.\n");
    } else {
        print("Moving $oldSkybotDir to $automateScheduleDir.\n");
        system("mv $oldSkybotDir $automateScheduleDir") == 0 || die("Unable to rename $oldSkybotDir to $automateScheduleDir\n");
    }
    createProductUser();
    if ($whoami eq 'root') {
        system("chown -R $automateUser:$skybotGID $automateScheduleDir") == 0 || die("Unable to change owner of $automateScheduleDir\n");
        system("/usr/sbin/userdel skybot") == 0 || print("Unable to delete 'skybot' user.\n");
    }
    
    # Set this here so it is correct when configAgentScripts is called.
    $javaDir = "$automateScheduleDir/jre";
    
    # Comment out since this needs to happen after copyFiles().
    # Configure the new runjava.sh JAVA_HOME to $automateScheduleDir/jre.
    # my $searchText = "=XXX";
    # my $scriptName = "runjava.sh";
    # my $javaDir = $installer->findJava();
    # $installer->copyScript("$fileDir/$scriptName", "$skybotAgentDir/$scriptName", $searchText, "=$javaDir") or die("\n"); 
}

sub configAgentScripts {
    my $scriptName = "runjava.sh";
    my $searchText = "XXX";
    $installer->copyScript("$fileDir/$scriptName", "$skybotAgentDir/$scriptName", $searchText, $javaDir) or fail('');
    
    # Now for the icjobrunner.sh
    my $scriptName = "icjobrunner.sh";
    $installer->copyScript("$fileDir/$scriptName", "$skybotAgentDir/$scriptName", $searchText, $javaDir) or fail('');
}

sub configInfo {
    chdir $skybotAgentDir || fail("Unable to change directory to $skybotAgentDir");

    system("./config.sh -f agent.xml -lp $tlPort -sa $serverAddress -sp $serverPort") == 0 ||
        fail("Unable to configure the agent: $!");

    my $config_file = "$skybotAgentDir/agent.xml";
    system("chmod 644 ${config_file}") == 0 || fail("Unable to change file permissions on ${config_file}: $!");

    chdir $startingDir;
}

sub blessNopwd {
    my $nopwd = "$skybotAgentDir/bin/$platform/nopwd";
    
    if ( (-f $nopwd) && ($whoami eq 'root') ) {
        system("chown root $nopwd") == 0 || carp("Unable to set root-ownership on $nopwd: $!");
        system("chmod 4750 $nopwd") == 0 || carp("Unable to setuid on $nopwd: $!");

    }
}

sub createSkybotUserForOSX {
    my $last_uid = 0;
    my @all_uid = `dscacheutil -q user | grep ^uid`;
    foreach my $line (@all_uid) {
        chomp($line);
        my ($ignored, $uid) = split /: /, $line;
        if ($uid > $last_uid) {
            $last_uid = $uid;
        }
    }
    if ($last_uid == 0) {
        return 0;
    }

    my $next_uid = $last_uid + 1;
    my $user = 'automate';
    my $home = $automateScheduleDir;

    system("dscl . -create /Users/$user") == 0 || return 0;
    system("dscl . -create /Users/$user UniqueID $next_uid") == 0 || return 0;
    system("dscl . -create /Users/$user UserShell /bin/sh") == 0 || return 0;
    system("dscl . -create /Users/$user RealName \"Automate Schedule User\"") == 0 || return 0;
    $skybotGID = 20;
    system("dscl . -create /Users/$user PrimaryGroupID $skybotGID") == 0 || return 0;
    system("dscl . -create /Users/$user NFSHomeDirectory $home") == 0 || return 0;

    system("mkdir -p $home") == 0 || return 0;
    system("chown $user $home") == 0 || return 0;
}

sub installJava {

    $installer->maybeDownload($javaInstallFile, \@remainingDownloads) || fail('');
    pop(@remainingDownloads);
    $installer->makeDir("$automateScheduleDir");
    chdir "$automateScheduleDir" or fail("Unable to change dir to $automateScheduleDir");
    $javaInstallFile = '"' . "$startingDir/$javaInstallFile" . '"';
    $javaInstallCmd =~ s/_FILE_/$javaInstallFile/g;
    print("Executing Java install command: $javaInstallCmd\n");
    system($javaInstallCmd) == 0 || fail("Java install failed.");
    if ($whoami eq 'root') {
        system("chown -R $automateUser:$skybotGID .") == 0 || fail("Unable to switch ownership of Automate Schedule Java files to $automateUser");
    }
    chdir $startingDir;
}

sub fail {
    my $message = shift;
    printf("$message\n");
    chdir $startingDir;
    exit 1;
}

sub showHelpAndExit {
    print("* To use an automatic Agent Install Configuration, please specify the URL\n");
    print("  of your Automate Schedule Server, like this:\n\n");
    print("         agentInstall https://yourAutomateScheduleServer:itsPort config_name\n\n");
    print("* The Automate Schedule Server info is usually the same that you use in your web browser.\n");
    print("* The 'config_name' is the name of the Agent Install Configuration.\n");
    print("  If 'config_name' is omitted, then 'Unix_default' is used.\n");
    exit(1);
}

sub testNativesz {
    my $javaExec = "$javaDir/bin/java";
    my $javaFlags = '';
    if ( ($os eq 'SunOS' && $bitWidth eq '64') || $os eq 'HP-UX') {
        $javaFlags = '-d64';
    }
    my $libs = new File::List("$fileDir/lib");
    my @files = @{ $libs->find("\.jar\$") };
    my $classpath;
    for (@files) {
        $classpath = $classpath . ":$_";
    }
    # debug lines
    #    print("Here are the details about my JVMTests call from installAgent.pl::testNativesz : \n");
    #    print("fileDir = $fileDir\n");
    #    print("javaExec = $javaExec\n");
    #    print("javaFlags = $javaFlags\n");
    #    print("LD_LIBRARY_PATH = $ENV{LD_LIBRARY_PATH}\n");
    #    print("classpath = $classpath\n");
    #    print("\n");
    my $exitCode = system("$javaExec $javaFlags -Djava.library.path=\$LD_LIBRARY_PATH -cp $classpath com.helpsystems.enterprise.install.JVMTests -n");
    if ($exitCode > 0) {
        fail("This usually occurs when the bit-width of Java does not match the kernel\n   (such as using 32-bit Java on a 64-bit system)\n");
   }
}

sub downloadExtraBinaries {

    if ( ($os eq 'SunOS') && ($rawArch eq 'i386') ) {
        # There isn't a SAP library for this combination
        print("Note: The SAP v3 native library is not available for 32-bit Solaris/Intel.\n");
        print("      SAP features are not available on this agent.\n");
        return 1;
    }

    my $localSAPLib = "$startingDir/$sapFile";
    my $agentSAPLib = "$skybotAgentDir/$sapLibPath";
    if ( ! -f $localSAPLib) {
        # see if the installed agent has it
        if ( -f "$agentSAPLib" ) {
            system("cp $agentSAPLib $localSAPLib");
        }
    }

    if ( ! -f $localSAPLib) {
        # cheat and see if the server has it
        my $serverSAPLib = "$startingDir/files-server/webapps/automate-schedule/WEB-INF/$sapLibPath";
        if ( -f $serverSAPLib) {
            system("cp $serverSAPLib $localSAPLib");
        }
    }
        
    if ( ! -f $localSAPLib) {
        # download it
        $installer->maybeDownload($sapLibPath, @remainingDownloads) || fail('');
    }

    system("cp $localSAPLib $startingDir/$fileDir/$sapLibPath") == 0 || fail(
        "Unable to copy the SAP library $localSAPLib to the installer dir");
}

sub copyFiles {

    print("Copying product files...\n");

    if( $os eq "AIX") {
        my $search = new File::List("$skybotAgentDir/bin/$platform");
        my @files = @{ $search->find("lib") };
        my $filecount = @files;
        if ($filecount > 0) {
            # Bug 13913: flush loaded shared libraries on AIX
            #   or not - /usr/sbin/slibclean can only be run as root.
            if ($whoami eq 'root') {
                system("/usr/sbin/slibclean");
            }

            # Otheriwse, a "well known" AIX "workaround" is to just rename it.
            for my $file (@files) {
                my $result = system("mv -f $file $skybotAgentDir/bin/$platform/trash");
                if ($result > 0) {
                    fail("Unable to remove $file");
                }
            }
        }
    }

    if ( -e "$skybotAgentDir/lib/sigarDlls") {
        system("rm -rf $skybotAgentDir/lib/trash > /dev/null 2> /dev/null");
        my $result = system("mv -f $skybotAgentDir/lib/sigarDlls $skybotAgentDir/lib/trash");
        if ($result > 0) {
            fail("Unable to remove $skybotAgentDir/lib/sigarDlls");
        }
    }

    # see if 'nopwd' is writable
    my $existingNopwd = "$skybotAgentDir/bin/$platform/nopwd";
    my $nopwdWritable = 1;
    if ( -f $existingNopwd && ! -w $existingNopwd ) {
        $nopwdWritable = 0;
        
        my $line = `sum $existingNopwd`;
        if ($? != 0) {
            fail("Unable to obtain the checksum of $existingNopwd");
        }
        chomp($line);
        my ($existingCksum, $ignored) = split / /, $line;

        my $newNopwd = "$fileDir/bin/$platform/nopwd";
        if (-f $newNopwd) {
            $line = `sum $newNopwd`;
            if ($? != 0) {
                fail("Unable to obtain the checksum of $newNopwd");
            }
            chomp($line);
            my ($newCksum, $ignored) = split / /, $line;

            if ($existingCksum == $newCksum) {
                # delete the new one, since it's the same as the existing one,
                #    and we won't be able to overwrite the existing one.
                system("rm -f $newNopwd") == 0 || fail("Unable to remove $newNopwd: $!");
            } else {
                fail("Unable to overwrite $existingNopwd, you may need 'root' to remove the file or run this installer.");
            }
        }
    }

    
    # remove obsolete/upgraded files during upgrade
    system("rm -f $skybotAgentDir/lib/httpclient-4.2.3.jar > /dev/null 2>&1");
    system("rm -f $skybotAgentDir/lib/httpcore-4.2.2.jar > /dev/null 2>&1");
    system("rm -f $skybotAgentDir/lib/commons-beanutils-1.8.2.jar > /dev/null 2>&1");
    system("rm -f $skybotAgentDir/lib/commons-codec-1.6.jar > /dev/null 2>&1");
    system("rm -f $skybotAgentDir/lib/commons-collections-3.2.1.jar > /dev/null 2>&1");
    

    system("cp -R -p $startingDir/$fileDir/* $skybotAgentDir") == 0 || fail("Unable to copy product files to $skybotAgentDir");

    # if we are on AIX, We need to create a symbolic link for the sigar library for AIX 7.1
    if ($os eq 'AIX') {
        print("Creating symbolic link for: $skybotAgentDir/bin/AIX/powerpc64/libsigar-ppc64-aix-7.so\n");
        my $sigar_lnk_result = system("ln -sf $skybotAgentDir/bin/AIX/powerpc64/libsigar-ppc64-aix-5.so $skybotAgentDir/bin/AIX/powerpc64/libsigar-ppc64-aix-7.so");
        if ($sigar_lnk_result > 0) {
           print("Error - Unable to create symbolic link sigar library.\n");
        }
    }

    if ($whoami eq 'root') {
        # change ownership of everything over to automate.
        system("chown -R $automateUser:$skybotGID $skybotAgentDir") == 0 || 
            fail("Unable to change ownership of $skybotAgentDir to $automateUser");
    }
    
    # Make sure everything is readable and executable
    system("chmod 755 $skybotAgentDir/*.sh") == 0 || 
        fail("Unable to set the file-executable rights on $skybotAgentDir/*.sh: $!");
    system("chmod -R 755 $skybotAgentDir/lib") == 0 || 
        fail("Unable to set the file-executable rights on $skybotAgentDir/lib: $!");

    if ($nopwdWritable) {
        system("chmod -R 755 $skybotAgentDir/bin") == 0 || 
            fail("Unable to set the file-executable rights on $skybotAgentDir/bin: $!");
    } else {
        my $fullbindir = "$skybotAgentDir/bin/$platform";
        opendir my $dir, "$fullbindir" or die "Cannot read dir $fullbindir: $!";
        my @files = readdir $dir;
        closedir $dir;
        for my $file (@files) {
            if (!($file =~ /nopwd/)) {
                my $fullpath = $fullbindir . '/' . $file;
                system("chmod 755 $fullpath") == 0 ||
                    fail("Unable to change file permissions on $fullpath: $!");
            }
        }
    }        
    
    my $joblogtmp = "$skybotAgentDir/joblogtmp";
    if ( ! -e $joblogtmp) {
        $installer->makeDir($joblogtmp);
        system("chmod 700 $joblogtmp") == 0 || 
        fail("Unable to set the permissions on $joblogtmp: $!");
    }

    # if we came from the Server install, then I need to run the execstack in case we are trying to use JRE7
    if ($os eq 'Linux') {
        # print("path= $skybotAgentDir/bin/Linux/$rawArch/libnativeSz.so\n");
        # print("execstack -c $skybotAgentDir/bin/Linux/$rawArch/libnativeSz.so\n");
        my $result = system("execstack -c $skybotAgentDir/bin/Linux/$rawArch/libnativeSz.so");
        if ($result > 0) {
           print("Warning only - Unable to mark nativeSz as not requiring an executable stack\n");
        }
    }
}

sub checkIfAgentRunning {
    my $file= "$skybotAgentDir/agent.xml.pid";
    if ( -r $file ) {
       my $OLD_PID=`cat ${file}`;
       chomp($OLD_PID);
       if ( $$ eq $OLD_PID) {
           # we're re-using that pid, that's fine.
           return 1;
       }
       my $PROCESS_COUNT=`ps -p ${OLD_PID} | grep ${OLD_PID} | wc -l`;
       if ( ${PROCESS_COUNT} > 0 ) {
            fail("Agent process $OLD_PID in $skybotAgentDir needs to be shutdown before installation can continue.");
        }
    }
}

sub checkIfOldSkybotAgentRunning {
    my $file= "$oldSkybotAgentDir/agent.xml.pid";
    if ( -r $file ) {
       my $OLD_PID=`cat ${file}`;
       chomp($OLD_PID);
       if ( $$ eq $OLD_PID) {
           # we're re-using that pid, that's fine.
           return 1;
       }
       my $PROCESS_COUNT=`ps -p ${OLD_PID} | grep ${OLD_PID} | wc -l`;
       if ( ${PROCESS_COUNT} > 0 ) {
            fail("Agent process $OLD_PID in $oldSkybotAgentDir needs to be shutdown before installation can continue.");
        }
    }
}

sub manualConfig {
    $serverAddress = $installer->prompt("Enter the name or IP address of the Automate Schedule Server");
    $serverPort = $installer->promptForPort("Enter the agent-connection port of the Automate Schedule Server", 7472, 0);
    $tlPort = $installer->promptForPort("Enter the port number to use for local connections", $tlPort);
}


# Clean up the installed file's attributes and Finish

if ($whoami eq 'root') {
    # change ownership of everything over to automate.
    system("chown -R $automateUser:$skybotGID $skybotAgentDir") == 0 || fail("Unable to change ownership of $skybotAgentDir to $automateUser");

    blessNopwd();
}

# If this is a rebrand
if ($rebrand == 1) {

    # Remove these files:
    system("rm -f $skybotAgentDir/skybot-import-cron.sh > /dev/null 2>&1");
    system("rm -f $skybotAgentDir/SkybotLicenseAgreement.html > /dev/null 2>&1");
    
    # And add these symlnks:
    # ln -sf {source} /path/to/file  {link} /path/to/symlink
    
    # Do not do this if the server has already.
    if ( -d $oldSkybotDir) {
        print("* Symlink already exists for $oldSkybotDir\n");
    } else {
        system("ln -sf $automateScheduleDir $oldSkybotDir > /dev/null 2>&1");
    }
    
    system("ln -sf $skybotAgentDir/import-cron.sh $oldSkybotAgentDir/skybot-import-cron.sh > /dev/null 2>&1");
    
}

system("chmod 755 $skybotAgentDir/*.sh") == 0 || 
    fail("Unable to set the file-executable rights on $skybotAgentDir/*.sh: $!");

my $command = "$skybotAgentDir/start.sh";
if ( $whoami eq 'root' ) {
    $command = "su $automateUser -c \"$command\"";
}

my $now_string_after = `date "+%Y-%m-%d %H:%m:%S"`;
print("Installation completed at $now_string_after \n");
if ($quietInstall == 0) {
    print("Do you want to start the agent now? [y] ");
}
my $done = 0;
while(! $done) {
    my $input;
    if ($quietInstall == 0) {
        $input = <STDIN>;
        chomp($input);
    } else {
        if ($startAgent) {
            $input = 'y';
        } else {
            $input = 'n';
        }
    }
    if ($input eq '' || $input eq 'y' || $input eq 'Y') {
        print("Starting agent with command: $command\n");
        my $fullCommandLine = "$command > /dev/null 2> /dev/null";

        if ($execAgent == 1) {
            # B: 5476 - don't exec fullCommandLine, because perl will
            #           insert an extra "sh -c" process which disrupts
            #           the existing authorized-agent entry.
            exec("$command");
            fail("Unable to start the agent: $!");
        } else {
            system($fullCommandLine . '&') == 0 || 
                fail("Unable to start the agent: $!");
        }
        $done = 1;
    } elsif ($input eq 'n' || $input eq 'N') {
        print("When you're ready to start the agent, use the command:\n");
        print("   $command\n");
        $done = 1;
    } else {
        print("Please answer Y or N: [y] ");
    }
}


if ($whoami eq 'root') {
    print("If you want the agent to start when this system boots up, add this command\n   to this system's startup sequence.\n");
}
